// Import CONFIG from config.js
import { CONFIG } from './config.js';
import { initializeSettings, syncSettings } from './settings-sync.js';

console.log("WHOIS Checker PRO background.js loaded");

// ========================
// REGISTER MESSAGE LISTENER IMMEDIATELY (before any async code)
// ========================
let messageListenerRegistered = false;
try {
    chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
        // Defer to processQueue/processQueueGptOnly after definition
        // IMPORTANT: Must return the result to keep message channel open for async responses
        return handleMessage(msg, sender, sendResponse);
    });
    messageListenerRegistered = true;
} catch (err) {
    console.error("[Background] Failed to register message listener:", err);
}

// Ensure side panel opens on action click
chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true })
    .catch((error) => console.error("Could not set panel behavior:", error));

// ========================
// Settings Synchronization
// ========================
// Initialize settings on extension startup (non-blocking)
// If this fails, the extension should still work
initializeSettings().catch((error) => {
    console.error('[Background] Settings initialization failed (non-blocking):', error);
    // Continue anyway - extension should work without settings sync
});

// Periodic settings sync (every hour)
const SETTINGS_SYNC_INTERVAL = 60 * 60 * 1000; // 1 hour
setInterval(() => {
    syncSettings(false).catch((error) => {
        console.error('[Background] Periodic settings sync failed:', error);
    });
}, SETTINGS_SYNC_INTERVAL);

// ========================
// Global Cache
// ========================
let whoisCache = {}; // domain -> whois result
let gptCache = {};   // domain -> { category, intro, intro_zh }

// Load cache from storage
chrome.storage.local.get(["whoisCache", "gptCache"], (res) => {
    if (res.whoisCache) {
        whoisCache = res.whoisCache;
        CONFIG.log("Loaded WHOIS cache:", Object.keys(whoisCache).length);
    }
    if (res.gptCache) {
        gptCache = res.gptCache;
        CONFIG.log("Loaded GPT cache:", Object.keys(gptCache).length);
    }
});

// Save cache helpers
function saveCache() {
    chrome.storage.local.set({ whoisCache });
}

function saveGptCache() {
    chrome.storage.local.set({ gptCache });
}

// ========================
// Delay & Timeout helpers
// ========================
function wait(ms) {
    return new Promise((resolve) => setTimeout(resolve, ms));
}

function fetchWithTimeout(url, options, timeout = 15000) {
    return Promise.race([
        fetch(url, options),
        new Promise((_, reject) =>
            setTimeout(() => reject(new Error('Request timeout')), timeout)
        )
    ]);
}

// ========================
// WHOIS Batch Request (via Backend Proxy)
// ========================
async function fetchWhoisBatch(domains, batchIndex) {
    const url = CONFIG.getBackendUrl(CONFIG.BACKEND.ENDPOINTS.WHOIS_PROXY);

    for (let attempt = 1; attempt <= 3; attempt++) {
        try {
            const res = await fetchWithTimeout(url, {
                method: "POST",
                headers: CONFIG.EXTENSION.getAuthHeaders(),
                body: JSON.stringify({
                    domains: domains,
                    source: "chrome-extension"
                })
            }, CONFIG.EXTENSION_REQUEST_TIMEOUT);

            // Handle 429 rate limit with longer delay
            if (res.status === 429) {
                const waitTime = Math.pow(2, attempt) * 1000; // 2s, 4s, 8s
                CONFIG.warn(`WHOIS Batch #${batchIndex} rate limited (429), retrying in ${waitTime/1000}s... (attempt ${attempt}/3)`);
                await wait(waitTime);
                continue;
            }

            if (!res.ok) {
                throw new Error(`Backend returned ${res.status}`);
            }

            const result = await res.json();

            // Check for both response formats:
            // - respData format: { code: 0, message: "ok", data: {...} }
            // - legacy format: { success: true, data: {...} }
            if ((result.code === 0 || result.success) && result.data) {
                CONFIG.log(`WHOIS Batch #${batchIndex} success`, result.data);
                return result.data; // Backend returns data in standard format
            } else {
                throw new Error(result.message || result.error || "WHOIS request failed");
            }

        } catch (err) {
            CONFIG.warn(`WHOIS Batch #${batchIndex} attempt ${attempt} failed`, err.message);
            if (attempt < 3) await wait(1000 * attempt);
        }
    }

    CONFIG.error(`WHOIS Batch #${batchIndex} failed after 3 retries`);
    return null;
}


// ========================
// GPT Analysis (via Backend Proxy)
// ========================
async function fetchGptBatch(domains, retryCount = 0) {
    if (domains.length === 0) return {};

    const url = CONFIG.getBackendUrl(CONFIG.BACKEND.ENDPOINTS.GPT_ANALYZE_PROXY);
    const maxRetries = 3;

    try {
        const res = await fetchWithTimeout(url, {
            method: "POST",
            headers: CONFIG.EXTENSION.getAuthHeaders(),
            body: JSON.stringify({
                domains: domains,
                source: "chrome-extension"
            })
        }, 40000); // 40 second timeout for GPT processing

        // Handle 429 rate limit with retry
        if (res.status === 429) {
            if (retryCount < maxRetries) {
                const waitTime = Math.pow(2, retryCount + 1) * 1000; // 2s, 4s, 8s
                CONFIG.warn(`GPT API rate limited (429), retrying in ${waitTime/1000}s... (attempt ${retryCount + 1}/${maxRetries})`);
                await wait(waitTime);
                return fetchGptBatch(domains, retryCount + 1);
            }
            CONFIG.error(`GPT API rate limited after ${maxRetries} retries`);
            return {};
        }

        if (!res.ok) {
            CONFIG.warn(`GPT API failed: ${res.status}`);
            return {}; // Return empty object, don't break the flow
        }

        const result = await res.json();

        // Backend returns: { code: 0, data: { data: { results: [...] } } }
        // Try both possible locations for results
        let resultsArray = null;

        if (result.code === 0 && result.data) {
            // respData format: nested data structure
            if (result.data.data && result.data.data.results) {
                resultsArray = result.data.data.results;
            }
            // Direct results in data (legacy or cache format)
            else if (result.data.results) {
                resultsArray = result.data.results;
            }
        } else if (result.success && result.data && result.data.results) {
            // Legacy format support
            resultsArray = result.data.results;
        }

        if (resultsArray) {
            // Convert backend response to extension format
            const gptData = {};
            resultsArray.forEach(item => {
                gptData[item.domain] = {
                    category: item.category,
                    intro: item.intro,
                    intro_zh: item.intro_zh
                };
            });
            CONFIG.log("GPT analysis complete", gptData);
            return gptData;
        }

        CONFIG.warn("GPT response did not match expected format", { code: result.code, success: result.success, dataStructure: result.data ? Object.keys(result.data) : 'no data' });
        return {};
    } catch (err) {
        CONFIG.error("GPT analysis error", err.message);
        return {};
    }
}

// ========================
// Queue Processor
// ========================
async function processQueue(domains, sendResponse, skipGpt = false) {
    // Always query the server for all domains - no local cache check
    const newDomains = domains;
    const gptMissing = skipGpt ? [] : domains;

    // Helper: Extract root domain with support for multi-part TLDs
    const getRootDomain = (hostname) => {
        const parts = hostname.split('.');
        if (parts.length <= 2) return hostname;

        // Common Multi-part TLDs & SLDs
        const multipartTLDs = [
            // UK
            'co.uk', 'org.uk', 'gov.uk', 'ac.uk', 'me.uk', 'ltd.uk', 'plc.uk', 'net.uk', 'sch.uk',
            // CN
            'com.cn', 'net.cn', 'org.cn', 'gov.cn', 'edu.cn', 'ac.cn', 'mil.cn',
            // AU
            'com.au', 'net.au', 'org.au', 'edu.au', 'gov.au', 'asn.au', 'id.au',
            // BR
            'com.br', 'net.br', 'org.br', 'gov.br', 'edu.br', 'mil.br', 'art.br',
            // JP
            'co.jp', 'ne.jp', 'or.jp', 'go.jp', 'ac.jp', 'ed.jp',
            // TW
            'com.tw', 'org.tw', 'net.tw', 'edu.tw', 'gov.tw',
            // HK
            'com.hk', 'org.hk', 'net.hk', 'edu.hk', 'gov.hk',
            // SG
            'com.sg', 'org.sg', 'net.sg', 'edu.sg', 'gov.sg',
            // NZ
            'co.nz', 'org.nz', 'net.nz', 'school.nz', 'ac.nz',
            // IL
            'co.il', 'org.il', 'net.il', 'ac.il', 'gov.il', 'muni.il',
            // Other Common
            'com.mx', 'com.tr', 'com.ru', 'com.de', 'com.fr'
        ];

        const last2 = parts.slice(-2).join('.');
        if (multipartTLDs.includes(last2) && parts.length >= 3) {
            return parts.slice(-3).join('.');
        }
        return parts.slice(-2).join('.');
    };

    // Always query the server - no early return for cached data

    // Define WHOIS worker
    const runWhois = async () => {
        if (newDomains.length === 0) {
            return;
        }

        // 1. Map original domains to root domains
        // domainMap: { "csrc.gov.cn": ["www.csrc.gov.cn"], "google.com": ["google.com"] }
        const domainMap = {};
        const rootBatches = new Set();

        newDomains.forEach(d => {
            const root = getRootDomain(d);
            if (!domainMap[root]) domainMap[root] = [];
            domainMap[root].push(d);
            rootBatches.add(root);
        });

        const uniqueRoots = Array.from(rootBatches);

        const batches = [];
        for (let i = 0; i < uniqueRoots.length; i += 20) {
            batches.push(uniqueRoots.slice(i, i + 20));
        }

        let batchNumber = 1;
        for (const batch of batches) {
            // Send progress update through port
            sendToPopup({
                type: "whoisProgress",
                message: `WHOIS Batch ${batchNumber}/${batches.length}...`
            });

            // Query the ROOT domains
            const data = await fetchWhoisBatch(batch, batchNumber);

            // data structure: { source, timestamp, requested, data: { results: [...] }, metadata }
            // Extract the actual WHOIS data from nested structure
            const whoisData = data?.data || data;
            const resultsArray = whoisData?.results;

            if (resultsArray) {
                // Handle results
                const resultsArrayFinal = Array.isArray(resultsArray) ? resultsArray : Object.values(resultsArray);

                // Send each domain result immediately for real-time rendering (like GPT does)
                const batchResults = {};
                resultsArrayFinal.forEach(item => {
                    const rootDomain = item.domain; // API returns root (e.g., csrc.gov.cn)

                    // Map result back to ALL original subdomains (e.g., www.csrc.gov.cn)
                    if (domainMap[rootDomain]) {
                        domainMap[rootDomain].forEach(originalDomain => {
                            whoisCache[originalDomain] = {
                                ...item,
                                _root: rootDomain // Debug info
                            };
                            batchResults[originalDomain] = whoisCache[originalDomain];
                        });
                    }

                    // Also cache the root domain itself
                    if (!whoisCache[rootDomain]) {
                        whoisCache[rootDomain] = item;
                        batchResults[rootDomain] = item;
                    }
                });

                saveCache();

                // Send incremental update with only this batch's results (for real-time rendering)
                if (Object.keys(batchResults).length > 0) {
                    sendToPopup({
                        type: "whoisBatchComplete",
                        results: batchResults
                    });
                    CONFIG.log(`Sent WHOIS batch ${batchNumber} update with ${Object.keys(batchResults).length} domains`);
                }
            }

            // Add delay between batches to avoid rate limiting (1s)
            if (batchNumber < batches.length) {
                await wait(1000);
            }

            batchNumber++;
        }
    };

    // Define GPT worker
    const runGpt = async () => {
        if (gptMissing.length === 0) return;

        const gptBatches = [];
        for (let i = 0; i < gptMissing.length; i += 5) {
            gptBatches.push(gptMissing.slice(i, i + 5));
        }

        let gptBatchNum = 1;
        for (const batch of gptBatches) {
            // Send progress update through port
            sendToPopup({
                type: "whoisProgress",
                message: `Analyzing Content ${gptBatchNum}/${gptBatches.length}...`
            });

            const results = await fetchGptBatch(batch);

            if (results && Object.keys(results).length > 0) {
                Object.keys(results).forEach(key => {
                    gptCache[key] = results[key];
                });
                saveGptCache();

                // Send incremental update through port
                sendToPopup({
                    type: "gptBatchComplete",
                    results: results
                });
                CONFIG.log(`Sent GPT batch ${gptBatchNum} update with ${Object.keys(results).length} domains`);
            }

            // Add delay between batches to avoid rate limiting (500ms)
            if (gptBatchNum < gptBatches.length) {
                await wait(500);
            }

            gptBatchNum++;
        };
    };

    // Run WHOIS first, then GPT sequentially to avoid rate limiting
    // This reduces 429 errors by spacing out requests to the backend
    try {
        await runWhois();
        // Add a small delay between WHOIS and GPT to avoid rate limiting
        await wait(200);
        await runGpt();
    } catch (err) {
        console.error("[processQueue] Queue processing error:", err);
    }

    // Sync data to backend database (don't block on this)
    syncWhoisToBackend(domains, whoisCache, gptCache).catch(err => {
        CONFIG.error("Background sync failed", err.message);
    });

    // Final completion message through port
    sendToPopup({
        type: "whoisProgress",
        message: "Analysis Completed."
    });

    // Ensure sendResponse is always called with valid data
    try {
        sendResponse({
            done: true,
            results: whoisCache || {},
            gptResults: gptCache || {}
        });
    } catch (err) {
        console.error("[processQueue] Error calling sendResponse:", err);
    }
}


// ========================
// GPT-Only Processor
// ========================
async function processQueueGptOnly(domains, sendResponse) {
    // Always query the server - no local cache check
    const gptMissing = domains;

    // Run GPT analysis only
    const gptBatches = [];
    for (let i = 0; i < gptMissing.length; i += 5) {
        gptBatches.push(gptMissing.slice(i, i + 5));
    }

    let gptBatchNum = 1;
    for (const batch of gptBatches) {
        // Send progress update through port
        sendToPopup({
            type: "whoisProgress",
            message: `Analyzing Content ${gptBatchNum}/${gptBatches.length}...`
        });

        const results = await fetchGptBatch(batch);

        if (results && Object.keys(results).length > 0) {
            Object.keys(results).forEach(key => {
                gptCache[key] = results[key];
            });
            saveGptCache();

            // Send incremental update through port
            sendToPopup({
                type: "gptBatchComplete",
                results: results
            });
            CONFIG.log(`Sent GPT-only batch ${gptBatchNum} update with ${Object.keys(results).length} domains`);
        }

        // Add delay between batches to avoid rate limiting (500ms)
        if (gptBatchNum < gptBatches.length) {
            await wait(500);
        }

        gptBatchNum++;
    }

    // Final completion message through port
    sendToPopup({
        type: "whoisProgress",
        message: "Analysis Completed."
    });

    sendResponse({
        done: true,
        gptResults: gptCache
    });
}


// ========================
// Message Port for Real-time Updates
// ========================
let popupPort = null;

chrome.runtime.onConnect.addListener((port) => {
    if (port.name === "whoisProcessor") {
        popupPort = port;
        CONFIG.log("Popup connected for real-time updates, port name:", port.name);

        port.onDisconnect.addListener(() => {
            popupPort = null;
            CONFIG.log("Popup disconnected");
        });
    }
});

// Helper: Send real-time update to popup
function sendToPopup(message) {
    if (popupPort) {
        try {
            popupPort.postMessage(message);
            CONFIG.log("Background: Message sent to popup -", message.type);
        } catch (err) {
            CONFIG.warn("Failed to send message to popup port", err.message);
        }
    } else {
        CONFIG.warn("Popup port not connected - message not sent:", message.type);
    }
}

// ========================
// Message Handler Function (called from registered listener above)
// ========================
function handleMessage(msg, sender, sendResponse) {
    if (msg.type === "queryWhoisFull") {
        // Send immediate acknowledgment to avoid Chrome timeout
        sendResponse({ acknowledging: true });
        // Process in background
        processQueue(msg.domains, () => {
            // Secondary response through port (already sent results via popupPort)
        }, true); // skipGpt = true - 禁用GPT分析
        return false; // Don't keep channel open - we already responded
    }
    if (msg.type === "queryWhoisOnly") {
        // Send immediate acknowledgment to avoid Chrome timeout
        sendResponse({ acknowledging: true });
        // Process in background
        processQueue(msg.domains, () => {
            // Secondary response through port (already sent results via popupPort)
        }, true); // skipGpt = true
        return false; // Don't keep channel open - we already responded
    }
    if (msg.type === "queryGptOnly") {
        // Send immediate acknowledgment to avoid Chrome timeout
        sendResponse({ acknowledging: true });
        // Process in background
        processQueueGptOnly(msg.domains, () => {
            // Secondary response through port (already sent results via popupPort)
        });
        return false; // Don't keep channel open - we already responded
    }
    if (msg.type === "clearCache") {
        if (msg.domains && Array.isArray(msg.domains)) {
            msg.domains.forEach(d => {
                delete whoisCache[d];
                delete gptCache[d];
            });
            saveCache();
            saveGptCache();
        }
        sendResponse({ success: true });
        return false;
    }
    if (msg.type === "clearWhoisCache") {
        if (msg.domains && Array.isArray(msg.domains)) {
            msg.domains.forEach(d => {
                delete whoisCache[d];
            });
            saveCache();
        }
        sendResponse({ success: true });
        return false;
    }

    // SimilarWeb Logic
    if (msg.type === "fetchSimilarWeb") {
        const domain = msg.domain;
        if (!domain) {
            sendResponse({ success: false, error: "No domain provided" });
            return false;
        }

        // Always fetch fresh data from backend - no caching
        fetchSimilarWeb(domain).then(result => {
            if (result) {
                // Send the full backend response to popup.js
                sendResponse(result);
            } else {
                sendResponse({ success: false });
            }
        });
        return true; // Async
    }

    // Auto-scan WHOIS query (from content.js)
    if (msg.type === "getWhoisAuto") {
        const domains = msg.domains || [];
        if (domains.length === 0) {
            sendResponse({ whoisData: {} });
            return false;
        }

        CONFIG.log(`Auto-scan WHOIS query for ${domains.length} domains`);

        // Query WHOIS for all domains
        (async () => {
            const whoisData = {};
            const batchSize = 20;

            for (let i = 0; i < domains.length; i += batchSize) {
                const batch = domains.slice(i, i + batchSize);
                const batchIndex = Math.floor(i / batchSize) + 1;

                // Check cache first
                const toQuery = batch.filter(d => !whoisCache[d]);

                if (toQuery.length > 0) {
                    CONFIG.log(`Fetching WHOIS batch ${batchIndex} for ${toQuery.length} domains:`, toQuery);
                    const data = await fetchWhoisBatch(toQuery, batchIndex);

                    // Extract results from nested structure: { source, timestamp, data: { results: [...] }, metadata }
                    const responseData = data?.data || data;
                    const resultsArray = responseData?.results;

                    if (resultsArray && Array.isArray(resultsArray)) {
                        CONFIG.log(`Got ${resultsArray.length} results from batch ${batchIndex}`);
                        resultsArray.forEach(item => {
                            if (item.domain) {
                                whoisCache[item.domain] = item;
                                whoisData[item.domain] = item;
                                CONFIG.log(`Cached WHOIS for ${item.domain}`);
                            }
                        });
                        saveCache();
                    } else {
                        CONFIG.warn(`Batch ${batchIndex} returned no results`, { data, responseData });
                    }
                } else {
                    CONFIG.log(`All domains in batch ${batchIndex} are cached`);
                }

                // Add cached results
                batch.forEach(d => {
                    if (whoisCache[d] && !whoisData[d]) {
                        whoisData[d] = whoisCache[d];
                    }
                });
            }

            CONFIG.log(`Auto-scan complete, returning ${Object.keys(whoisData).length} domains`);
            sendResponse({ whoisData });
        })();

        return true; // Keep channel open for async response
    }

    // Settings sync request (from popup or content script)
    if (msg.type === "syncSettings") {
        const force = msg.force || false;
        CONFIG.log(`Settings sync requested (force: ${force})`);

        syncSettings(force).then((settings) => {
            sendResponse({ success: true, settings: settings });
        }).catch((error) => {
            CONFIG.error("Settings sync failed:", error);
            sendResponse({ success: false, error: error.message });
        });

        return true; // Keep channel open for async response
    }

    return false;
}

// ========================
// SimilarWeb Fetcher (via Backend Proxy)
// ========================
let simWebCache = {};
chrome.storage.local.get(["simWebCache"], (res) => {
    if (res.simWebCache) simWebCache = res.simWebCache;
});

function saveSimWebCache() {
    chrome.storage.local.set({ simWebCache });
}

async function fetchSimilarWeb(domain) {
    const url = CONFIG.getBackendUrl(CONFIG.BACKEND.ENDPOINTS.SIMILARWEB_PROXY);

    try {
        const res = await fetchWithTimeout(url, {
            method: "POST",
            headers: CONFIG.EXTENSION.getAuthHeaders(),
            body: JSON.stringify({
                domain: domain,
                source: "chrome-extension"
            })
        }, CONFIG.EXTENSION_REQUEST_TIMEOUT);

        if (!res.ok) {
            CONFIG.error(`SimilarWeb API error: ${res.status}`);
            return null;
        }

        const result = await res.json();
        CONFIG.log(`SimilarWeb API raw response for ${domain}:`, result);

        // Backend returns: { code: 0, data: { domain, source, data: {...} } }
        // We want to return the entire response to popup.js for proper handling
        if (result.code === 0 && result.data) {
            CONFIG.log(`SimilarWeb data for ${domain} - returning full response`);
            // Note: Database sync is already handled by /api/seo/proxy/similarweb endpoint
            // Return the full backend response structure
            return result;
        }

        return null;
    } catch (err) {
        CONFIG.error("SimilarWeb fetch failed", err.message);
        return null;
    }
}

// ========================
// Sync Data to Backend Database
// ========================
async function syncWhoisToBackend(domains, whoisData, gptData) {
    if (domains.length === 0) return;

    const url = CONFIG.getBackendUrl(CONFIG.BACKEND.ENDPOINTS.SYNC_WHOIS);

    try {
        const payload = {
            domains: domains.map(domain => {
                const domainWhoisInfo = whoisData[domain]?.result?.domain || {};

                // Safely parse domainStatus
                let domainStatus = [];
                if (Array.isArray(domainWhoisInfo.domain_status)) {
                    domainStatus = domainWhoisInfo.domain_status;
                } else if (typeof domainWhoisInfo.status === 'string') {
                    domainStatus = domainWhoisInfo.status.split(',').map(s => s.trim());
                }

                // Safely parse nameServers
                let nameServers = [];
                if (Array.isArray(domainWhoisInfo.name_servers)) {
                    nameServers = domainWhoisInfo.name_servers;
                } else if (typeof domainWhoisInfo.nameservers === 'string') {
                    nameServers = domainWhoisInfo.nameservers.split(',').map(s => s.trim());
                }

                return {
                    domain: domain,
                    // Date fields
                    registeredAt: domainWhoisInfo.created_date,
                    expiresAt: domainWhoisInfo.expiration_date,
                    whoisUpdatedAt: domainWhoisInfo.updated_date,
                    whoisFetchedAt: new Date().toISOString(),
                    // Basic info
                    homepageUrl: domainWhoisInfo.homepage,
                    siteTitle: domainWhoisInfo.title,
                    // WHOIS details
                    registrar: domainWhoisInfo.registrar,
                    registrarId: domainWhoisInfo.registrar_id,
                    domainStatus: domainStatus,
                    nameServers: nameServers,
                    dnssec: domainWhoisInfo.dnssec,
                    // Metadata
                    language: domainWhoisInfo.language,
                    countryCode: domainWhoisInfo.country,
                    category: gptData[domain]?.category,
                    intro_zh: gptData[domain]?.intro_zh,
                    source: "chrome-extension"
                }
            })
        };

        const res = await fetch(url, {
            method: 'POST',
            headers: CONFIG.EXTENSION.getAuthHeaders(),
            body: JSON.stringify(payload)
        });

        if (res.ok) {
            const result = await res.json();
            CONFIG.log(`Synced ${result.data?.synced || domains.length} domains to backend`);
        } else {
            CONFIG.warn(`Sync failed: ${res.status}`);
        }
    } catch (err) {
        CONFIG.error("Sync to backend failed", err.message);
    }
}

