/**
 * Chrome Extension Configuration
 *
 * ⚠️ IMPORTANT: This file should NEVER contain API keys or external API URLs.
 * All external API calls MUST go through the backend proxy layer.
 *
 * Architecture:
 *   Chrome Extension → Backend API Gateway → External APIs
 *                       (隐藏所有密钥)
 *
 * This ensures:
 * ✅ API keys are secure (stored in backend .env only)
 * ✅ Keys can be updated without redeploying the extension
 * ✅ API usage can be monitored and rate-limited
 * ✅ Easier to implement caching and optimization
 */

export const CONFIG = {
  // ===========================
  // Chrome Extension Authentication
  // ===========================
  // Unified extension ID used across all environments and channels
  // This ensures the same validation logic regardless of deployment
  EXTENSION: {
    // Unified ID shared between frontend and backend
    // Used to identify all requests from this extension
    ID: "websitetrafficchecker",

    // Secret key shared with backend for signature generation
    // Used across all environments (dev, staging, production)
    API_SECRET_KEY: "GLwKSviSVskbm+ZGj0NE6o/2xGOQytUC2yBJmX3/Ysg=",

    /**
     * Get authentication headers for API requests
     *
     * Uses unified ID to ensure consistent validation across all deployments.
     * The actual chrome.runtime.id is not used - we use a fixed application ID instead.
     *
     * @returns {Object} - Headers object with authentication
     */
    getAuthHeaders() {
      const timestamp = Date.now();
      const extensionId = CONFIG.EXTENSION.ID;  // Use unified ID
      const secretKey = CONFIG.EXTENSION.API_SECRET_KEY;

      // Generate signature
      const message = `${timestamp}_${extensionId}_${secretKey}`;
      const signature = btoa(unescape(encodeURIComponent(message)));

      return {
        'Content-Type': 'application/json',
        'X-Extension-ID': extensionId,
        'X-Timestamp': timestamp.toString(),
        'X-Signature': signature,
        'X-Source': 'chrome-extension'
      };
    }
  },

  // ===========================
  // Backend API Gateway Configuration
  // ===========================
  // All external API calls are proxied through these endpoints
  BACKEND: {
    // 开发环境: http://localhost:3000
    // 生产环境: https://websitetrafficchecker.net
    APP_URL: "https://websitetrafficchecker.net",

    ENDPOINTS: {
      // WHOIS API 代理
      WHOIS_PROXY: "/api/seo/proxy/whois",

      // GPT 分析代理
      GPT_ANALYZE_PROXY: "/api/seo/proxy/gpt-analyze",

      // SimilarWeb 数据代理（会自动同步到数据库）
      SIMILARWEB_PROXY: "/api/seo/proxy/similarweb",

      // 同步 WHOIS 数据到数据库
      SYNC_WHOIS: "/api/seo/fetch/sync-whois",

      // Chrome Extension Settings
      SETTINGS_PROXY: "/api/seo/proxy/settings",
    }
  },

  // ===========================
  // Extension Behavior Configuration
  // ===========================
  // Note: Cache and timeout settings are added below to EXTENSION object

  // Cache settings (managed by chrome.storage)
  EXTENSION_CACHE_WHOIS: true,        // 缓存WHOIS查询结果
  EXTENSION_CACHE_GPT: true,          // 缓存GPT分析结果
  EXTENSION_CACHE_SIMILARWEB: true,   // 缓存SimilarWeb数据

  // Request timeout (milliseconds)
  EXTENSION_REQUEST_TIMEOUT: 30000,   // 30秒超时（后端处理）

  // Batch processing
  EXTENSION_BATCH_SIZE: 20,           // 批处理域名数
  EXTENSION_BATCH_DELAY: 1000,        // 批次间隔 (ms)

  // ===========================
  // Logging Configuration
  // ===========================
  DEBUG: true,

  // ===========================
  // Utility Methods
  // ===========================

  /**
   * 构建后端API完整URL
   */
  getBackendUrl(endpoint) {
    return this.BACKEND.APP_URL + endpoint;
  },

  /**
   * 获取代理API端点
   */
  getProxyEndpoint(type) {
    const map = {
      'whois': this.BACKEND.ENDPOINTS.WHOIS_PROXY,
      'gpt': this.BACKEND.ENDPOINTS.GPT_ANALYZE_PROXY,
      'similarweb': this.BACKEND.ENDPOINTS.SIMILARWEB_PROXY,
      'sync': this.BACKEND.ENDPOINTS.SYNC_WHOIS
    };
    return map[type] || null;
  },

  /**
   * Make authenticated API request to backend
   * @param {string} endpoint - API endpoint
   * @param {Object} data - Request body data
   * @returns {Promise} - Fetch response
   */
  async makeApiRequest(endpoint, data) {
    const url = this.getBackendUrl(endpoint);
    const headers = this.EXTENSION.getAuthHeaders();

    this.log(`Making API request to: ${url}`, { data });

    try {
      const response = await fetch(url, {
        method: 'POST',
        headers: headers,
        body: JSON.stringify(data)
      });

      if (!response.ok) {
        const errorText = await response.text();
        this.error(`API request failed: ${response.status} ${response.statusText}`, errorText);
        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
      }

      const result = await response.json();
      this.log(`API response received`, result);
      return result;
    } catch (error) {
      this.error('Network error during API request', error);
      throw error;
    }
  },

  /**
   * 调试日志
   */
  log(message, data = null) {
    if (this.DEBUG) {
      console.log(`[SEO-WHOIS] ${message}`, data || "");
    }
  },

  /**
   * 警告日志
   */
  warn(message, data = null) {
    console.warn(`[SEO-WHOIS] WARNING: ${message}`, data || "");
  },

  /**
   * 错误日志
   */
  error(message, data = null) {
    console.error(`[SEO-WHOIS] ERROR: ${message}`, data || "");
  },
};
