console.log("WHOIS Checker PRO content.js loaded");

// =============================================
// Global configuration (will be loaded from server or use defaults)
// =============================================
let globalDomainConfig = {
    allowedSuffixes: null,
    configLoadTime: null,
    isUsingLocalFallback: false
};

// =============================================
// Utility: Extract age from created_date
// =============================================
function calculateAge(createdDate) {
    if (!createdDate) return "";
    const cd = new Date(createdDate);
    const now = new Date();

    const diffMs = now - cd;
    const days = Math.floor(diffMs / (1000 * 60 * 60 * 24));
    const months = Math.floor(days / 30);
    const years = Math.floor(months / 12);

    if (years > 0) return `${years}y ${months % 12}m`;
    if (months > 0) return `${months} months`;
    return `${days} days`;
}

// =============================================
// Utility: Risk color
// =============================================
function getRiskColor(createdDate) {
    if (!createdDate) return "gray";

    const cd = new Date(createdDate);
    const now = new Date();

    const diffDays = Math.floor((now - cd) / (1000 * 60 * 60 * 24));

    if (diffDays < 90) return "red";   // < 3 months
    if (diffDays < 365) return "gold"; // < 12 months
    return "gray";                     // older
}

// =============================================
// Inject Tooltip CSS
// =============================================
(function injectTooltipStyles() {
    const css = `
    .whois-tag {
        font-size: 10px;
        margin-left: 4px;
        cursor: default;
    }
    .whois-tooltip {
        position: absolute;
        background: #222;
        padding: 4px 6px;
        color: #fff;
        border-radius: 4px;
        font-size: 10px;
        display: none;
        z-index: 999999;
    }
    `;
    const style = document.createElement("style");
    style.textContent = css;
    document.head.appendChild(style);
})();

// Tooltip element
const tooltip = document.createElement("div");
tooltip.className = "whois-tooltip";
document.body.appendChild(tooltip);

// =============================================
// Extract domain columns (text only, not links)
// =============================================
// Local fallback whitelist of allowed domain suffixes (TLDs and second-level domains)
// This will be used if server config cannot be loaded
const LOCAL_ALLOWED_SUFFIXES = [
    // Generic TLDs
    'com', 'org', 'net', 'info', 'biz', 'name', 'pro', 'co',
    'edu', 'gov', 'mil', 'int',
    // New gTLDs - Technology
    'tech', 'dev', 'design', 'art', 'law', 'med', 'health',
    'finance', 'bank', 'insurance', 'legal', 'engineer', 'architect',
    'doctor', 'nurse', 'teacher', 'school', 'university', 'college',
    'academy', 'cloud', 'data', 'ai', 'ml', 'dl', 'robot', 'iot',
    'app', 'software', 'hardware', 'game', 'gamer', 'stream', 'video',
    'music', 'film', 'photo', 'studio', 'media',
    // Lifestyle & Services
    'travel', 'food', 'cafe', 'restaurant', 'hotel', 'resort',
    'fit', 'fitness', 'yoga', 'sport', 'pet', 'pets',
    'book', 'books', 'read', 'writer', 'author',
    'fashion', 'style', 'beauty', 'cosmetics', 'jewelry', 'watch',
    'car', 'auto', 'bike', 'cycle',
    'home', 'house', 'realty', 'property', 'garden', 'flower',
    // Internet Infrastructure
    'online', 'site', 'space', 'store', 'shop', 'market',
    'sale', 'deal', 'offer', 'service', 'services', 'support', 'help',
    'solution', 'tool', 'tools', 'web', 'website', 'page',
    'blog', 'vlog', 'log', 'note', 'notes',
    // Country TLDs and second-level domains
    'cn', 'com.cn', 'net.cn', 'org.cn', 'gov.cn', 'edu.cn',
    'hk', 'mo', 'tw',
    'us', 'uk', 'co.uk', 'org.uk',
    'jp', 'co.jp',
    'kr', 'co.kr',
    'au', 'com.au',
    'ca', 'com.ca',
    'de', 'fr', 'it', 'es', 'ru', 'br', 'in',
    'me', 'tv', 'cc', 'io', 'ws',
    'eu', 'asia', 'africa', 'latin', 'arab',
    'sg', 'my', 'th', 'vn', 'ph', 'id',
    // Specialty domains
    'museum', 'post', 'tel', 'mobi', 'jobs', 'aero', 'coop',
    'xxx', 'sucks', 'adult', 'family', 'kids',
    'church', 'mosque', 'temple',
    'army', 'navy', 'airforce', 'police', 'fire', 'emergency'
];

// =============================================
// Load domain configuration from chrome.storage.local
// =============================================
async function loadDomainConfig() {
    console.log('WHOIS Checker: Loading domain configuration');

    try {
        // First, try to load from chrome.storage.local
        const result = await new Promise((resolve) => {
            chrome.storage.local.get(['domainSettings'], (res) => {
                resolve(res.domainSettings);
            });
        });

        if (result && result.allowedSuffixes && Array.isArray(result.allowedSuffixes)) {
            globalDomainConfig.allowedSuffixes = new Set(result.allowedSuffixes);
            globalDomainConfig.configLoadTime = new Date();
            globalDomainConfig.isUsingLocalFallback = false;
            console.log("WHOIS Checker: Loaded domain config from storage", {
                version: result.version,
                lastUpdated: result.lastUpdated,
                suffixCount: result.allowedSuffixes.length,
                loadTime: globalDomainConfig.configLoadTime
            });
            return;
        }

        // If no stored settings, trigger sync in background
        console.log("WHOIS Checker: No stored settings found, requesting sync");
        chrome.runtime.sendMessage({ type: "syncSettings", force: false }, (response) => {
            // Check if settings actually has content (not just empty object)
            const hasSettings = response && response.success && response.settings &&
                Object.keys(response.settings).length > 0;

            if (hasSettings) {
                console.log("WHOIS Checker: Settings synced, reloading config");
                loadDomainConfig(); // Reload after sync
            } else {
                console.warn("WHOIS Checker: Sync returned no settings, using local fallback");
                useFallbackConfig();
            }
        });

    } catch (error) {
        console.error("WHOIS Checker: Error loading config:", error);
        useFallbackConfig();
    }
}

// Fallback to local configuration
function useFallbackConfig() {
    globalDomainConfig.allowedSuffixes = new Set(LOCAL_ALLOWED_SUFFIXES);
    globalDomainConfig.configLoadTime = new Date();
    globalDomainConfig.isUsingLocalFallback = true;
    console.log("WHOIS Checker: Using local fallback domain config", {
        suffixCount: LOCAL_ALLOWED_SUFFIXES.length,
        loadTime: globalDomainConfig.configLoadTime
    });
}

// Initialize config on script load
loadDomainConfig();

// Listen for settings updates from background
chrome.storage.onChanged.addListener((changes, areaName) => {
    if (areaName === 'local' && changes.domainSettings) {
        console.log("WHOIS Checker: Settings updated, reloading config");
        loadDomainConfig();
    }
});

// Helper function to check if domain suffix is in whitelist
function isDomainAllowed(domain) {
    // Use global config, fallback to local list if not yet loaded
    const allowedSet = globalDomainConfig.allowedSuffixes || new Set(LOCAL_ALLOWED_SUFFIXES);

    const parts = domain.split('.');
    if (parts.length < 2) return false;

    // Check multi-part suffixes first (e.g., co.uk, com.cn)
    if (parts.length >= 3) {
        const twoPartSuffix = parts.slice(-2).join('.');
        if (allowedSet.has(twoPartSuffix)) {
            return true;
        }
    }

    // Check single-part suffix (TLD)
    const tld = parts[parts.length - 1];
    return allowedSet.has(tld);
}

// List of uncommon gTLDs that are often false positives when appearing as X.tld
// These are real TLDs but rarely appear as actual domains with single-letter subdomains
const UNCOMMON_GTLDS = new Set([
    'key', 'submit', 'shiftkey', 'mov', 'feedback', 'bio', 'club',
    'consulting', 'dance', 'date', 'dental', 'design', 'dev', 'digital', 'discount',
    'education', 'email', 'engineering', 'equipment', 'estate', 'events', 'expert',
    'fail', 'farm', 'finance', 'financial', 'fitness', 'flights', 'florist', 'flowers',
    'fly', 'football', 'forum', 'foundation', 'fun', 'fund', 'furniture', 'fyi',
    'gallery', 'garden', 'gas', 'gdn', 'general', 'genetics', 'geographic', 'gift',
    'gifts', 'giveaway', 'giving', 'glass', 'global', 'globe', 'gmail', 'gmbh',
    'gmo', 'gmx', 'gold', 'golf', 'goodhands', 'google', 'gop', 'got', 'gov',
    'gown', 'gp', 'gpage', 'gq', 'gr', 'grad', 'graduateway', 'grainger'
]);

// Common legitimate TLDs (less likely to be false positives)
const COMMON_TLDS = new Set([
    'com', 'org', 'net', 'edu', 'gov', 'mil', 'int', 'biz', 'info', 'name',
    'co', 'uk', 'us', 'cn', 'de', 'fr', 'it', 'es', 'nl', 'be', 'ch', 'at',
    'se', 'no', 'dk', 'fi', 'pl', 'ru', 'jp', 'kr', 'au', 'nz', 'in', 'br',
    'mx', 'ca', 'ie', 'za', 'hk', 'sg', 'tw', 'th', 'vn', 'id', 'ph'
]);

// 获取节点的直接文本内容（不包含子元素的文本）
function getDirectTextContent(node) {
    let text = '';
    for (const child of node.childNodes) {
        if (child.nodeType === Node.TEXT_NODE) {
            text += child.textContent;
        }
    }
    return text;
}

function extractDomainTexts() {
    console.log("WHOIS Checker: Starting domain extraction...");
    const nodes = document.querySelectorAll(
        // 基础文本容器
        "td, th, div, span, p, a, " +
        // 标题标签
        "h1, h2, h3, h4, h5, h6, " +
        // 列表相关
        "li, dt, dd, " +
        // 文本格式标签
        "strong, em, b, i, u, mark, code, pre, " +
        // 引用和代码
        "blockquote, cite, q, samp, kbd, var, " +
        // 表单相关
        "label, button, " +
        // 语义化标签
        "article, section, aside, header, footer, nav, main, " +
        // 其他常用标签
        "figcaption, caption, summary, address, time, small"
    );
    const domains = new Set();

    // Improved regex to match common domain patterns
    const regex = /\b([a-zA-Z0-9]([a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?\.)+[a-zA-Z]{2,}\b/g;

    nodes.forEach(node => {
        // 只获取直接文本内容，避免嵌套提取
        // 例如 <div>abc<a>d.com</a></div> 只从div提取"abc"，从a提取"d.com"
        let text = getDirectTextContent(node).trim();

        // Skip very long text blocks and empty nodes
        if (!text || text.length > 300) return;

        let match;
        // Reset regex lastIndex
        regex.lastIndex = 0;

        while ((match = regex.exec(text)) !== null) {
            let domain = match[0].trim().toLowerCase();

            // Filter out common false positives
            if (domain.length < 4) continue;
            if (domain.endsWith('.png') || domain.endsWith('.jpg') ||
                domain.endsWith('.jpeg') || domain.endsWith('.gif') ||
                domain.endsWith('.svg') || domain.endsWith('.css') ||
                domain.endsWith('.js')) continue;

            // Must have at least one dot and valid TLD
            const parts = domain.split('.');
            if (parts.length < 2) continue;

            const tld = parts[parts.length - 1];
            if (tld.length < 2 || tld.length > 10) continue;

            // Check if domain suffix is in the whitelist
            if (!isDomainAllowed(domain)) {
                console.log(`Filtered: ${domain} (suffix not in whitelist)`);
                continue;
            }

            const subdomain = parts[0];

            // Strategy: Filter single-character subdomains unless it's a COMMON TLD
            // x.com, x.org, x.net etc. are kept (common TLDs)
            // x.key, x.submit etc. are filtered (uncommon gTLDs)
            if (subdomain.length === 1) {
                if (!COMMON_TLDS.has(tld)) {
                    console.log(`Filtered: ${domain} (single-char subdomain with non-common TLD)`);
                    continue;
                }
            }
            // For 2-char subdomains, be extra careful with uncommon gTLDs
            else if (subdomain.length === 2 && UNCOMMON_GTLDS.has(tld)) {
                console.log(`Filtered: ${domain} (short subdomain with uncommon gTLD)`);
                continue;
            }

            domains.add(domain);
        }
    });

    const result = Array.from(domains);
    console.log(`WHOIS Checker: Found ${result.length} domains:`, result.slice(0, 10));
    return result;
}

// =============================================
// Apply WHOIS results to webpage
// =============================================
function applyWhoisToPage(results) {
    const allDomains = Object.keys(results);

    allDomains.forEach(domain => {
        const whois = results[domain];
        const cd = whois?.result?.domain?.created_date?.slice(0, 10) || "";
        const color = getRiskColor(cd);
        const ageStr = calculateAge(cd);

        // Find matching text nodes
        const xpath = `//*[contains(text(), '${domain}')]`;
        const nodes = document.evaluate(xpath, document, null, XPathResult.ORDERED_NODE_SNAPSHOT_TYPE, null);

        for (let i = 0; i < nodes.snapshotLength; i++) {
            const el = nodes.snapshotItem(i);

            // Skip if already appended
            if (el.dataset.whoisApplied === "1") continue;

            const html = el.innerHTML;

            // Create replacement tag
            const tag = `<span class="whois-tag" style="color:${color}" data-domain="${domain}" data-created="${cd}">
                            (${cd || "N/A"})
                         </span>`;

            const newHTML = html.replace(domain, domain + " " + tag);
            el.innerHTML = newHTML;
            el.dataset.whoisApplied = "1";
        }
    });

    // Attach tooltip listeners
    document.querySelectorAll(".whois-tag").forEach(tag => {
        tag.addEventListener("mouseenter", e => {
            const cd = tag.dataset.created;
            tooltip.innerText = cd ? `Age: ${calculateAge(cd)}` : "No WHOIS data";
            tooltip.style.display = "block";
            tooltip.style.top = (e.pageY + 10) + "px";
            tooltip.style.left = (e.pageX + 10) + "px";
        });

        tag.addEventListener("mouseleave", () => {
            tooltip.style.display = "none";
        });
    });
}

// =============================================
// Track domains we've already processed
// =============================================
let processedDomains = new Set();
let domObserverInitialized = false;

// =============================================
// Monitor DOM for dynamically loaded content
// =============================================
let domObserverTimeout = null;

function debouncedDomCheck() {
    // Clear previous timeout if it exists
    if (domObserverTimeout) {
        clearTimeout(domObserverTimeout);
    }

    // Set a new timeout to check for new domains
    domObserverTimeout = setTimeout(() => {
        const newDomains = extractDomainTexts();
        const unprocessedDomains = newDomains.filter(d => !processedDomains.has(d));

        if (unprocessedDomains.length > 0) {
            console.log(`WHOIS Checker: Detected ${unprocessedDomains.length} new domains in dynamic content:`, unprocessedDomains.slice(0, 5));

            // Request WHOIS data for new domains
            chrome.runtime.sendMessage(
                { type: "getWhoisAuto", domains: unprocessedDomains },
                (response) => {
                    if (chrome.runtime.lastError) {
                        console.error("WHOIS Checker: Error getting WHOIS data for dynamic content:", chrome.runtime.lastError);
                        return;
                    }

                    if (response && response.whoisData && Object.keys(response.whoisData).length > 0) {
                        console.log(`WHOIS Checker: Received WHOIS data for ${Object.keys(response.whoisData).length} new domains`);
                        applyWhoisToPage(response.whoisData);
                        console.log("WHOIS Checker: Applied new domains to page");
                    }
                }
            );

            // Mark these domains as processed
            unprocessedDomains.forEach(d => processedDomains.add(d));
        }

        domObserverTimeout = null;
    }, 500); // Wait 500ms after DOM changes before checking for new domains
}

function initDOMObserver() {
    if (domObserverInitialized) {
        console.log("WHOIS Checker: DOM observer already initialized, skipping...");
        return;
    }

    console.log("WHOIS Checker: Initializing DOM observer for dynamic content...");

    const observer = new MutationObserver((mutations) => {
        // Use debouncing to avoid excessive domain extraction on rapid DOM changes
        debouncedDomCheck();
    });

    // Observe changes to the document
    observer.observe(document.documentElement, {
        childList: true,      // Watch for added/removed nodes
        subtree: true,        // Watch all descendants
        characterData: false  // Don't watch text content changes (too noisy)
    });

    domObserverInitialized = true;
    console.log("WHOIS Checker: DOM observer initialized");
}

// =============================================
// Message Handlers
// =============================================
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
    if (msg.type === "scanDomains") {
        const domains = extractDomainTexts();

        // Mark these initial domains as processed and init observer
        domains.forEach(d => processedDomains.add(d));
        if (!domObserverInitialized) {
            initDOMObserver();
        }

        sendResponse({ domains });
        return true;
    }

    if (msg.type === "applyWhois") {
        applyWhoisToPage(msg.results);
        sendResponse({ done: true });
        return true;
    }


    return false;
});
