console.log("WHOIS Checker PRO popup.js loaded");

let scannedDomains = [];
let whoisData = {};
let gptData = {};
let simWebCache = {}; // SimilarWeb data cache
let statusBox = null;
let statusText = null;
let countBadge = null;
let currentLanguage = "en"; // Default language

// ============================================
// Translations
// ============================================
const translations = {
    en: {
        // Tab labels
        currentScan: "Current Scan",
        history: "History",

        // Buttons
        scanPage: "Scan Page",
        queryWhois: "Query WHOIS",
        exportCsv: "Export CSV",
        clear: "Clear",
        addDomains: "Add Domains",
        addDomainsManual: "Add Domains",
        submit: "Add Domains",
        cancel: "Cancel",
        previous: "Previous",
        next: "Next",

        // Manual input
        enterDomains: "Enter one domain per line (e.g., example.com)",
        addDomainsManually: "Add Domains Manually",

        // Search placeholders
        searchDomains: "Search domains...",
        searchHistory: "Search history...",

        // Status messages
        readyToScan: "Ready to scan",
        copying: (text) => `Copied "${text}"`,
        copyFailed: "Copy failed.",

        // Table headers
        domain: "Domain",
        category: "Category",
        intro: "Intro",
        created: "Created",
        age: "Age",

        // Empty states
        noDomains: "No domains found yet.",
        loadingHistory: "Loading history...",

        // History buttons
        filter: "Filter",
        clearWhois: "Clear WHOIS",
        exportAll: "Export All",

        // Modal headers
        analyzeTraffic: "Analyzing Traffic Data...",
        domainCreation: "Domain Creation",
        domainExpiration: "Domain Expiration",
        lastChanged: "Last Changed",
        domainAge: "Domain Age",
        totalVisits: "Total Visits",
        avgDuration: "Avg. Duration",
        pagesPerVisit: "Pages / Visit",
        bounceRate: "Bounce Rate",
        visitsOverTime: "Visits Over Time",
        trafficSources: "Traffic Sources",
        topRegions: "Top Regions",
        topKeywords: "Top Keywords",

        // Page info
        pageInfo: (page) => `Page ${page}`,

        // Domain count
        domains: (count) => `${count} Domains`,
        formattedDomains: (count) => `${count} Domains`,
    },
    zh: {
        // Tab labels
        currentScan: "当前扫描",
        history: "历史记录",

        // Buttons
        scanPage: "扫描页面",
        queryWhois: "查询WHOIS",
        exportCsv: "导出CSV",
        clear: "清除",
        addDomains: "添加域名",
        addDomainsManual: "添加域名",
        submit: "添加域名",
        cancel: "取消",
        previous: "上一页",
        next: "下一页",

        // Manual input
        enterDomains: "每行输入一个域名 (例如: example.com)",
        addDomainsManually: "手动添加域名",

        // Search placeholders
        searchDomains: "搜索域名...",
        searchHistory: "搜索历史...",

        // Status messages
        readyToScan: "准备扫描",
        copying: (text) => `已复制 "${text}"`,
        copyFailed: "复制失败。",

        // Table headers
        domain: "域名",
        category: "分类",
        intro: "介绍",
        created: "创建日期",
        age: "年龄",

        // Empty states
        noDomains: "还没有发现域名。",
        loadingHistory: "正在加载历史...",

        // History buttons
        filter: "筛选",
        clearWhois: "清除WHOIS",
        exportAll: "导出全部",

        // Modal headers
        analyzeTraffic: "分析流量数据中...",
        domainCreation: "域名创建",
        domainExpiration: "域名过期",
        lastChanged: "最后更改",
        domainAge: "域名年龄",
        totalVisits: "总访问次数",
        avgDuration: "平均停留时间",
        pagesPerVisit: "每次访问页数",
        bounceRate: "跳出率",
        visitsOverTime: "访问趋势",
        trafficSources: "流量来源",
        topRegions: "热门地区",
        topKeywords: "热门关键词",

        // Page info
        pageInfo: (page) => `第 ${page} 页`,

        // Domain count
        domains: (count) => `${count} 个域名`,
        formattedDomains: (count) => `${count} 个域名`,
    }
};

// Get translation text
function t(key) {
    const lang = translations[currentLanguage] || translations.en;
    return lang[key] !== undefined ? lang[key] : translations.en[key];
}

// ============================================
// Utility: Domain Structure Validator
// ============================================
/**
 * Validate domain structure: must contain at least one dot
 * Minimal validation to avoid false positives on unusual domains
 * Valid: example.com, xx-xx.com, xx_xx.com, sub.example.co.uk, etc.
 * Invalid: example (no dot), example. (trailing dot), .example (leading dot)
 * @param {string} domain - Domain to validate
 * @returns {boolean} - True if valid
 */
function validateDomainStructure(domain) {
    if (!domain || typeof domain !== 'string') return false;

    // Must contain at least one dot - this is the key requirement
    if (!domain.includes('.')) return false;

    // Must not start or end with a dot
    if (domain.startsWith('.') || domain.endsWith('.')) return false;

    // Must not have consecutive dots
    if (domain.includes('..')) return false;

    // That's it - minimal validation to avoid false positives
    return true;
}

// ============================================
// Utility: Date Normalizer
// ============================================
function normalizeDate(dateStr) {
    if (!dateStr) return null;

    // Filter out obvious noise or partial fragments like "before Aug"
    if (dateStr.length < 6 || dateStr.toLowerCase().includes("before")) return null;

    // 1. Try ISO-like or standard formats first
    let d = new Date(dateStr);
    if (!isNaN(d.getTime())) {
        return d.toISOString().split('T')[0];
    }

    // 2. Handle "2024. 02. 04" or "2024.02.04"
    if (/^\d{4}[\.\s]+\d{2}[\.\s]+\d{2}.*$/.test(dateStr)) {
        return dateStr.replace(/[\.\s]+/g, '-').slice(0, 10);
    }

    // 3. Handle "YYYYMMDD" (e.g., 20250106)
    if (/^\d{8}$/.test(dateStr)) {
        return `${dateStr.slice(0, 4)}-${dateStr.slice(4, 6)}-${dateStr.slice(6, 8)}`;
    }

    // 4. Handle "30-May-200" (Truncated year 2000? or 2001? Assume 20xx if 200)
    // Regex for DD-MMM-YY or DD-MMM-YYYY
    // e.g. "30-May-201", "31-May-200"
    const montMap = {
        jan: '01', feb: '02', mar: '03', apr: '04', may: '05', jun: '06',
        jul: '07', aug: '08', sep: '09', oct: '10', nov: '11', dec: '12'
    };

    const parts = dateStr.match(/^(\d{1,2})[-/]([A-Za-z]{3})[-/](\d{2,4})/);
    if (parts) {
        let day = parts[1].padStart(2, '0');
        let mon = montMap[parts[2].toLowerCase()];
        let year = parts[3];

        if (year.length === 3) year = year + "0"; // Guessing "200" -> "2000"? Highly ambiguous.
        // Based on image "31-May-200", it looks like truncated 200x. Let's assume 20 + last digit
        // But "29-Sep-202" -> 2020?
        // Let's rely on user verification for weird data, but try to fix standard "2001"
        if (year.length === 4) return `${year}-${mon}-${day}`;
    }

    // 5. "Tue Jun 28" (Missing year) -> return as is or ignore?
    // Returning null for now to avoid showing misleading dates (like 2001 default)
    return null;
}

// ============================================
// Utility: Calculate age (Robust)
// ============================================
function calculateAge(createdDate) {
    if (!createdDate) return "";
    const cleanDate = normalizeDate(createdDate);
    if (!cleanDate) return "";

    const cd = new Date(cleanDate);
    const now = new Date();

    const diffMs = now - cd;
    const days = Math.floor(diffMs / (1000 * 60 * 60 * 24));

    if (days < 0) return "0 days"; // Future date?

    const months = Math.floor(days / 30);
    const years = Math.floor(months / 12);

    if (years > 0) return `${years}y ${months % 12}m`;
    if (months > 0) return `${months} months`;
    return `${days} days`;
}

// ============================================
// Risk color
// ============================================
function getRiskLevel(created) {
    if (!created) return "low";
    const cd = new Date(created);
    const now = new Date();
    const diff = Math.floor((now - cd) / (1000 * 60 * 60 * 24));

    if (diff < 90) return "high";
    if (diff < 365) return "medium";
    return "low";
}

// ============================================
// Status updater
// ============================================
function setStatus(msg, type = "idle") {
    if (!statusText || !statusBox) return;
    statusText.textContent = msg;
    statusBox.className = "status-box";
    statusBox.classList.add(type);
}

// ============================================
// Language Management
// ============================================
function initLanguage() {
    chrome.storage.local.get(["language"], (res) => {
        if (res.language) {
            currentLanguage = res.language;
        } else {
            currentLanguage = "en";
            chrome.storage.local.set({ language: "en" });
        }
        updateLanguageUI();
        loadState();
    });
}

function setLanguage(lang) {
    if (translations[lang]) {
        currentLanguage = lang;
        chrome.storage.local.set({ language: lang });
        updateLanguageUI();
        updateAllUIText();
    }
}

function updateLanguageUI() {
    const langToggle = document.getElementById("langToggle");
    if (langToggle) {
        langToggle.value = currentLanguage;
    }
}

function updateAllUIText() {
    // Update tab buttons
    const tabScan = document.getElementById("tabScan");
    const tabHistory = document.getElementById("tabHistory");
    if (tabScan) tabScan.textContent = t("currentScan");
    if (tabHistory) tabHistory.textContent = t("history");

    // Update buttons
    const btnScan = document.getElementById("btnScan");
    const btnQueryWhois = document.getElementById("btnQueryWhois");
    const btnQueryAnalyze = document.getElementById("btnQueryAnalyze");
    const btnQueryFull = document.getElementById("btnQueryFull");
    const btnExport = document.getElementById("btnExport");
    const btnClear = document.getElementById("btnClear");
    const btnAddManual = document.getElementById("btnAddManual");
    const btnSubmitManual = document.getElementById("btnSubmitManual");
    const btnCancelManual = document.getElementById("btnCancelManual");
    const btnHistSearch = document.getElementById("btnHistSearch");
    const btnHistExport = document.getElementById("btnHistExport");
    const btnHistClearWhois = document.getElementById("btnHistClearWhois");
    const btnPrev = document.getElementById("btnPrev");
    const btnNext = document.getElementById("btnNext");

    if (btnScan) btnScan.innerHTML = `<span class="icon">🔍</span> ${t("scanPage")}`;
    if (btnQueryWhois) btnQueryWhois.textContent = t("queryWhois");
    if (btnQueryAnalyze) btnQueryAnalyze.textContent = "Analyze";
    if (btnQueryFull) btnQueryFull.textContent = "Query + Analyze";
    if (btnExport) btnExport.textContent = t("exportCsv");
    if (btnClear) btnClear.textContent = t("clear");
    if (btnAddManual) btnAddManual.innerHTML = `<span class="icon">➕</span> ${t("addDomains")}`;
    if (btnSubmitManual) btnSubmitManual.textContent = t("submit");
    if (btnCancelManual) btnCancelManual.textContent = t("cancel");
    if (btnHistSearch) btnHistSearch.textContent = t("filter");
    if (btnHistExport) btnHistExport.textContent = t("exportAll");
    if (btnHistClearWhois) btnHistClearWhois.textContent = t("clearWhois");
    if (btnPrev) btnPrev.textContent = t("previous");
    if (btnNext) btnNext.textContent = t("next");

    // Update search placeholders
    const scanSearch = document.getElementById("scanSearch");
    const histSearch = document.getElementById("histSearch");
    if (scanSearch) scanSearch.placeholder = t("searchDomains");
    if (histSearch) histSearch.placeholder = t("searchHistory");

    // Update manual input box
    const manualDomainInput = document.getElementById("manualDomainInput");
    const manualInputHeader = document.querySelector(".manual-input-header h3");
    const manualInputDesc = document.querySelector(".manual-input-header p");
    if (manualInputHeader) manualInputHeader.textContent = t("addDomainsManually");
    if (manualInputDesc) manualInputDesc.textContent = t("enterDomains");
    if (manualDomainInput) manualDomainInput.placeholder = "example.com\ngoogle.com\ngithub.com";

    // Update table headers
    const domainHeader = document.querySelector('th[data-key="domain"]');
    const categoryHeader = document.querySelector('th[data-key="category"]');
    const introHeader = document.querySelector('th[data-key="intro"]');
    const createdHeader = document.querySelector('th[data-key="created"]');
    const ageHeader = document.querySelector('th[data-key="age"]');

    if (domainHeader) domainHeader.textContent = t("domain");
    if (categoryHeader) categoryHeader.textContent = t("category");
    if (introHeader) introHeader.textContent = t("intro");
    if (createdHeader) createdHeader.textContent = t("created");
    if (ageHeader) ageHeader.textContent = t("age");

    // Update empty states
    const emptyState = document.getElementById("emptyState");
    const histEmpty = document.getElementById("histEmpty");
    if (emptyState) emptyState.textContent = t("noDomains");
    if (histEmpty) histEmpty.textContent = t("loadingHistory");

    // Update status
    setStatus(t("readyToScan"));

    // Re-render list to update domain count
    renderList();
}

// ============================================
// Data Persistence
// ============================================
function saveState() {
    chrome.storage.local.set({
        scannedDomains,
        whoisData,
        gptData,
        simWebCache  // Also save SimilarWeb cache for persistence
    });
}

function loadState() {
    chrome.storage.local.get(["scannedDomains", "whoisData", "gptData", "simWebCache"], (res) => {
        scannedDomains = res.scannedDomains || [];
        whoisData = res.whoisData || {};
        gptData = res.gptData || {};

        // Load and normalize simWebCache - transform snake_case to PascalCase if needed
        const rawSimWebCache = res.simWebCache || {};
        simWebCache = {};
        Object.keys(rawSimWebCache).forEach(domain => {
            const cached = rawSimWebCache[domain];
            if (cached && cached.data) {
                // Check if data needs transformation (has snake_case keys)
                if (cached.data.global_rank !== undefined || cached.data.monthly_visits !== undefined) {
                    simWebCache[domain] = {
                        timestamp: cached.timestamp,
                        data: transformBackendToFrontend(cached.data)
                    };
                } else {
                    simWebCache[domain] = cached;
                }
            }
        });

        renderList();
    });
}

// ============================================
// Sort Logic
// ============================================
// State
let currentSortKey = "none";
let currentSortDir = "asc"; // asc, desc

function handleSort(key) {
    if (currentSortKey === key) {
        // Toggle direction
        currentSortDir = currentSortDir === "asc" ? "desc" : "asc";
    } else {
        // New key
        currentSortKey = key;
        currentSortDir = "asc";
    }
    updateSortUI();
    renderList();
}

function updateSortUI() {
    const ths = document.querySelectorAll("th.sortable");
    ths.forEach(th => {
        th.classList.remove("asc", "desc");
        if (th.dataset.key === currentSortKey) {
            th.classList.add(currentSortDir);
        }
    });
}

function getSortedDomains() {
    if (currentSortKey === "none") return scannedDomains;

    return [...scannedDomains].sort((a, b) => {
        const itemA = whoisData[a] || {};
        const gptA = gptData[a] || {};
        const itemB = whoisData[b] || {};
        const gptB = gptData[b] || {};

        let valA, valB;

        switch (currentSortKey) {
            case "domain":
                valA = a.toLowerCase();
                valB = b.toLowerCase();
                break;
            case "category":
                valA = (gptA.category || "").toLowerCase();
                valB = (gptB.category || "").toLowerCase();
                break;
            case "intro":
                valA = (currentLanguage === "zh" ? gptA.intro_zh : (gptA.intro || gptA.intro_zh)) || "";
                valB = (currentLanguage === "zh" ? gptB.intro_zh : (gptB.intro || gptB.intro_zh)) || "";
                break;
            case "created":
            case "age":
                const dateA = itemA.result?.domain?.created_date;
                const dateB = itemB.result?.domain?.created_date;
                valA = dateA ? new Date(dateA).getTime() : 0;
                valB = dateB ? new Date(dateB).getTime() : 0;
                break;
        }

        if (valA < valB) return currentSortDir === "asc" ? -1 : 1;
        if (valA > valB) return currentSortDir === "asc" ? 1 : -1;
        return 0;
    });
}

// ============================================
// Render List (Table View)
// ============================================
function renderList() {
    const tableBody = document.getElementById("tableBody");
    const emptyState = document.getElementById("emptyState");

    // Safety check just in case DOM isn't ready
    if (!tableBody || !emptyState) return;

    tableBody.innerHTML = "";
    if (countBadge) {
        const count = scannedDomains.length;
        const countText = count >= 1000 ? `${formatNumber(count)}` : `${count}`;
        countBadge.textContent = currentLanguage === "zh" ? `${countText} 个域名` : `${countText} Domains`;
    }

    if (scannedDomains.length === 0) {
        emptyState.style.display = "block";
        return;
    } else {
        emptyState.style.display = "none";
    }

    const domainsToRender = getSortedDomains();

    // V1.0 SIMPLIFIED: Only show Domain and Created columns
    domainsToRender.forEach((domain, index) => {
        const tr = document.createElement("tr");
        tr.dataset.domain = domain;

        // Staggered animation
        tr.style.opacity = "0";
        tr.style.animation = `slideIn 0.3s ease forwards ${index * 0.03}s`;

        const item = whoisData[domain];
        const created = item?.result?.domain?.created_date?.slice(0, 10) || "";
        const risk = getRiskLevel(created);

        // Determine risk class for color coding
        let riskClass = "domain-low";
        if (risk === "high") riskClass = "domain-high";
        if (risk === "medium") riskClass = "domain-medium";

        // 1. Domain column (with color coding)
        const tdDomain = document.createElement("td");
        tdDomain.className = "col-domain";

        // Create domain name span with color coding
        const domainSpan = document.createElement("span");
        domainSpan.className = `domain-name ${riskClass}`;
        domainSpan.textContent = domain;
        domainSpan.style.cursor = "pointer";
        domainSpan.title = "Click to copy";
        domainSpan.onclick = () => copyToClipboard(domain);

        tdDomain.appendChild(domainSpan);

        // Add external link icon
        const link = document.createElement("a");
        link.href = `https://websitetrafficchecker.net/domain/${domain}`;
        link.target = "_blank";
        link.title = "View detailed traffic analysis";
        link.style.marginLeft = "8px";
        link.style.display = "inline-flex";
        link.style.alignItems = "center";
        link.style.color = "#94a3b8"; // text-muted color
        link.style.transition = "color 0.2s";

        link.onmouseover = () => { link.style.color = "#3b82f6"; }; // primary color
        link.onmouseout = () => { link.style.color = "#94a3b8"; };

        link.innerHTML = `<svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"></path>
            <polyline points="15 3 21 3 21 9"></polyline>
            <line x1="10" y1="14" x2="21" y2="3"></line>
        </svg>`;

        // Create a wrapper for proper alignment
        const wrapper = document.createElement("div");
        wrapper.style.display = "flex";
        wrapper.style.alignItems = "center";
        wrapper.style.justifyContent = "space-between";

        wrapper.appendChild(domainSpan);
        wrapper.appendChild(link);

        tdDomain.appendChild(wrapper);
        tr.appendChild(tdDomain);

        // 2. Intro column (language-aware)
        const tdIntro = document.createElement("td");
        tdIntro.className = "col-intro";
        const gptInfo = gptData[domain] || {};
        const introText = currentLanguage === "zh"
            ? (gptInfo.intro_zh || gptInfo.intro)
            : (gptInfo.intro || gptInfo.intro_zh);

        if (introText) {
            const introDiv = document.createElement("div");
            introDiv.className = "domain-intro";
            introDiv.textContent = introText;
            introDiv.dataset.tooltip = introText;
            tdIntro.appendChild(introDiv);
        } else {
            tdIntro.innerHTML = `<span style="color:#ccc;">-</span>`;
        }
        tr.appendChild(tdIntro);

        // 3. Created Date column
        const tdDate = document.createElement("td");
        tdDate.className = "col-date";
        tdDate.textContent = created || "-";
        // Apply color based on risk level
        if (risk === "high") {
            tdDate.style.color = "var(--risk-high)";
        } else if (risk === "medium") {
            tdDate.style.color = "var(--risk-med)";
        } else {
            tdDate.style.color = "var(--text-muted)";
        }
        tr.appendChild(tdDate);

        tableBody.appendChild(tr);
    });
}

// ============================================
// Seamless Row Update (No Full Re-render)
// ============================================
function updateRowData(domain, type = "all") {
    const tableBody = document.getElementById("tableBody");
    if (!tableBody) {
        console.log(`updateRowData: tableBody not found for ${domain}`);
        return;
    }

    const row = tableBody.querySelector(`tr[data-domain="${domain}"]`);
    if (!row) {
        console.log(`updateRowData: row not found for ${domain}, searching in:`, tableBody.innerHTML.substring(0, 200));
        return; // Row not found, skip
    }

    console.log(`updateRowData: found row for ${domain}, updating...`);
    const item = whoisData[domain];
    const rawCreated = item?.result?.domain?.created_date || "";
    console.log(`updateRowData: ${domain} - rawCreated:`, rawCreated);
    const created = normalizeDate(rawCreated) || "";
    console.log(`updateRowData: ${domain} - normalizeDate result:`, created);
    const risk = getRiskLevel(created);
    const gptInfo = gptData[domain] || {};

    // Update WHOIS data (Created Date and Age columns)
    if (type === "all" || type === "whois") {
        // Update domain risk class
        const domainCell = row.querySelector(".col-domain .domain-name");
        if (domainCell) {
            let riskClass = "domain-low";
            if (risk === "high") riskClass = "domain-high";
            if (risk === "medium") riskClass = "domain-medium";
            domainCell.className = `domain-name ${riskClass}`;
        }

        // Update Created Date
        const dateCell = row.querySelector(".col-date");
        if (dateCell) {
            dateCell.textContent = created || "-";
            // Apply color based on risk level
            if (risk === "high") {
                dateCell.style.color = "var(--risk-high)";
            } else if (risk === "medium") {
                dateCell.style.color = "var(--risk-med)";
            } else {
                dateCell.style.color = "var(--text-muted)";
            }
            console.log(`updateRowData: ${domain} - updated dateCell to "${created}"`);
        } else {
            console.log(`updateRowData: ${domain} - dateCell not found`);
        }

        // Update Age
        const ageCell = row.querySelector(".col-age");
        if (ageCell) {
            ageCell.innerHTML = "";
            if (created) {
                const ageText = calculateAge(created);
                const ageBadge = document.createElement("span");
                ageBadge.className = `age-badge risk-${risk}`;
                ageBadge.textContent = ageText;
                ageCell.appendChild(ageBadge);
                console.log(`updateRowData: ${domain} - updated ageCell to "${ageText}"`);
            } else {
                ageCell.textContent = "-";
                console.log(`updateRowData: ${domain} - no created date, set ageCell to "-"`);
            }
        } else {
            console.log(`updateRowData: ${domain} - ageCell not found`);
        }
    }

    // Update GPT data (Category and Intro columns)
    if (type === "all" || type === "gpt") {
        // Update Category
        const catCell = row.querySelector(".col-cat");
        if (catCell) {
            catCell.innerHTML = "";
            if (gptInfo.category) {
                const catBadge = document.createElement("span");
                catBadge.className = "category-badge";
                catBadge.textContent = gptInfo.category;
                catCell.appendChild(catBadge);
            } else {
                catCell.innerHTML = `<span style="color:#ccc;">-</span>`;
            }
        }

        // Update Intro (language-aware)
        const introCell = row.querySelector(".col-intro");
        if (introCell) {
            introCell.innerHTML = "";
            const introText = currentLanguage === "zh"
                ? (gptInfo.intro_zh || gptInfo.intro)
                : (gptInfo.intro || gptInfo.intro_zh);
            if (introText) {
                const introDiv = document.createElement("div");
                introDiv.className = "domain-intro";
                introDiv.textContent = introText;
                introDiv.dataset.tooltip = introText;
                introCell.appendChild(introDiv);
            } else {
                introCell.innerHTML = `<span style="color:#ccc;">-</span>`;
            }
        }
    }
}

// ============================================
// Manual Domain Input
// ============================================
function toggleManualInput() {
    const inputBox = document.getElementById("manualInputBox");
    const textarea = document.getElementById("manualDomainInput");

    if (inputBox.style.display === "none") {
        inputBox.style.display = "block";
        textarea.focus();
    } else {
        inputBox.style.display = "none";
        textarea.value = ""; // Clear input
    }
}

function processManualDomains() {
    const textarea = document.getElementById("manualDomainInput");
    const input = textarea.value.trim();

    if (!input) {
        setStatus("Please enter at least one domain.", "error");
        return;
    }

    // Split by newlines and filter out empty lines
    const lines = input.split('\n').map(line => line.trim()).filter(line => line.length > 0);

    // Extract domains using regex (same pattern as content.js)
    const domainRegex = /\b([a-zA-Z0-9]([a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?\.)+[a-zA-Z]{2,}\b/g;
    const extractedDomains = new Set();
    const invalidDomains = [];

    lines.forEach(line => {
        const matches = line.match(domainRegex);
        if (matches) {
            matches.forEach(domain => {
                // Basic validation
                const cleanDomain = domain.toLowerCase().trim();

                // Enhanced validation: must have xxx.xx structure (at least one dot and 2-char TLD)
                if (!validateDomainStructure(cleanDomain)) {
                    invalidDomains.push(cleanDomain);
                    return;
                }

                if (cleanDomain.length >= 4) {
                    // Filter out file extensions
                    if (!cleanDomain.endsWith('.png') && !cleanDomain.endsWith('.jpg') &&
                        !cleanDomain.endsWith('.jpeg') && !cleanDomain.endsWith('.gif') &&
                        !cleanDomain.endsWith('.svg') && !cleanDomain.endsWith('.css') &&
                        !cleanDomain.endsWith('.js')) {
                        extractedDomains.add(cleanDomain);
                    }
                }
            });
        }
    });

    if (invalidDomains.length > 0) {
        setStatus(`Invalid domain format: ${invalidDomains.slice(0, 3).join(', ')}${invalidDomains.length > 3 ? '...' : ''}. Domains must have xxx.xx structure.`, "error");
        return;
    }

    if (extractedDomains.size === 0) {
        setStatus("No valid domains found. Please check your input.", "error");
        return;
    }

    // Merge with existing domains (avoid duplicates)
    const newDomains = Array.from(extractedDomains).filter(d => !scannedDomains.includes(d));

    if (newDomains.length === 0) {
        setStatus("All domains already exist in the list.", "idle");
        toggleManualInput();
        return;
    }

    // Add new domains to the list
    scannedDomains = [...scannedDomains, ...newDomains];

    // Sync domains to backend database (fire-and-forget)
    syncDomainsToBackend(newDomains).catch(err => {
        console.error('Failed to sync domains to backend:', err);
    });

    // Always fetch fresh data from server - no cache restoration
    setStatus(`Added ${newDomains.length} new domains. Click "Query WHOIS" to fetch data.`, "success");
    saveState();
    renderList();
    toggleManualInput();
}

/**
 * Sync manually added domains to backend database
 */
async function syncDomainsToBackend(domains) {
    if (!domains || domains.length === 0) return;

    try {
        // Import CONFIG dynamically (it's loaded in the page context)

        // 开发环境: http://localhost:3000
        // 生产环境: https://websitetrafficchecker.net
        const backendUrl = "https://websitetrafficchecker.net"; // Should match config.js
        const url = `${backendUrl}/api/seo/fetch/sync-whois`;

        const payload = {
            domains: domains.map(domain => ({
                domain: domain,
                homepageUrl: `https://${domain}`,
                source: "chrome-extension-manual"
            }))
        };

        const res = await fetch(url, {
            method: 'POST',
            headers: CONFIG.EXTENSION.getAuthHeaders(),
            body: JSON.stringify(payload)
        });

        if (res.ok) {
            const result = await res.json();
            console.log(`Synced ${result.data?.synced || domains.length} manually added domains to backend`);
        } else {
            console.warn(`Domain sync failed: ${res.status}`);
        }
    } catch (err) {
        console.error("Sync domains to backend failed", err);
    }
}

// ============================================
// Core: Scan Domains
// ============================================
function scanDomains() {
    setStatus("Scanning page…", "scanning");

    chrome.tabs.query({ active: true, currentWindow: true }, tabs => {
        if (!tabs || !tabs[0]) {
            setStatus("No active tab found.", "error");
            return;
        }

        const url = tabs[0].url;
        if (!url || url.startsWith('chrome://') || url.startsWith('edge://') || url.startsWith('about:')) {
            setStatus("Cannot scan internal pages.", "error");
            return;
        }

        // Content script is auto-injected via manifest, just send message
        chrome.tabs.sendMessage(tabs[0].id, { type: "scanDomains" }, (res) => {
            if (chrome.runtime.lastError) {
                // Content script not ready yet, wait and retry once
                console.log("Content script not ready, retrying...");
                setTimeout(() => {
                    chrome.tabs.sendMessage(tabs[0].id, { type: "scanDomains" }, (res2) => {
                        if (chrome.runtime.lastError) {
                            setStatus("Scanner not ready. Please refresh the page and try again.", "error");
                            console.error("Scanner error:", chrome.runtime.lastError);
                            return;
                        }
                        handleScanResult(res2);
                    });
                }, 500);
            } else {
                handleScanResult(res);
            }
        });
    });
}

function handleScanResult(res) {
    if (!res || !res.domains) {
        setStatus("Scan failed or no domains found.", "error");
        return;
    }
    scannedDomains = Array.from(new Set(res.domains));
    setStatus(`Found ${scannedDomains.length} domains. Querying WHOIS + Analysis…`, "scanning");

    // Clear previous data - always fetch fresh from server
    whoisData = {};
    gptData = {};

    saveState();
    renderList();

    // AUTO: Immediately trigger WHOIS + GPT analysis after scan completes
    setTimeout(() => {
        queryFull();
    }, 300);
}

// ============================================
// Message Port Management for Real-time Updates
// ============================================
let processorPort = null;

function connectToProcessor() {
    // Establish persistent connection for real-time batch updates
    // Note: Always create a fresh connection to avoid stale port references
    // Previous connections may appear valid but actually be disconnected
    if (processorPort) {
        try {
            // Check if the old port is still valid by attempting to post a test message
            // This will throw an error if the port is disconnected
            // Actually, we can't reliably check port status without errors
            // So we'll close the old one and create a new connection
            processorPort = null; // Reset to force new connection
        } catch (e) {
            processorPort = null;
        }
    }

    processorPort = chrome.runtime.connect({ name: "whoisProcessor" });

    processorPort.onMessage.addListener((msg) => {

        if (msg.type === "whoisProgress") {
            setStatus(msg.message, "scanning");
        }
        if (msg.type === "whoisBatchComplete") {
            if (msg.results) {
                // Seamless update: only update changed domains
                const updatedDomains = [];
                Object.keys(msg.results).forEach(domain => {
                    if (!whoisData[domain] || JSON.stringify(whoisData[domain]) !== JSON.stringify(msg.results[domain])) {
                        whoisData[domain] = msg.results[domain];
                        updatedDomains.push(domain);
                    }
                });
                saveState();
                // Update only changed rows immediately for real-time rendering
                updatedDomains.forEach(domain => {
                    updateRowData(domain, "whois");
                });

                // NEW: Apply results to webpage immediately per batch (not waiting for all batches to complete)
                chrome.tabs.query({ active: true, currentWindow: true }, tabs => {
                    if (tabs[0]) {
                        chrome.tabs.sendMessage(tabs[0].id, {
                            type: "applyWhois",
                            results: msg.results  // Apply only this batch's results
                        });
                    }
                });
            }
        }
        if (msg.type === "gptBatchComplete") {
            // Seamless update: only update new GPT results
            if (msg.results) {
                Object.keys(msg.results).forEach(d => {
                    gptData[d] = msg.results[d];
                    updateRowData(d, "gpt"); // Update individual row immediately
                });
                saveState(); // Save progressively
            }
        }
    });

    processorPort.onDisconnect.addListener(() => {
        processorPort = null;
    });
}

// ============================================
// Core: Query WHOIS Only
// ============================================
function queryWhoisOnly() {
    if (scannedDomains.length === 0) {
        setStatus("No domains to query.", "idle");
        return;
    }

    setStatus("Querying WHOIS…", "scanning");

    // Establish connection for real-time updates
    connectToProcessor();

    try {
        chrome.runtime.sendMessage({ type: "queryWhoisOnly", domains: scannedDomains }, (res) => {
            if (chrome.runtime.lastError) {
                // Request may still be processing via popupPort
                return;
            }

            // Handle immediate acknowledgment response
            if (res && res.acknowledging) {
                // Results will arrive via popupPort's onMessage listener
                return;
            }

            // Handle complete response (legacy path - shouldn't happen with new code)
            if (!res || !res.done) {
                setStatus("Request failed.", "error");
                return;
            }

            whoisData = res.results || {};
            setStatus("WHOIS query completed!", "success");
            saveState();
        });
    } catch (error) {
        console.error("Error calling chrome.runtime.sendMessage:", error);
        setStatus("Error: " + error.message, "error");
    }
}


// ============================================
// Query Analyze Only (GPT Analysis)
// ============================================
function queryAnalyzeOnly() {
    if (scannedDomains.length === 0) {
        setStatus("No domains to analyze.", "idle");
        return;
    }

    setStatus("Starting Analysis…", "scanning");

    // Establish connection for real-time updates
    connectToProcessor();

    chrome.runtime.sendMessage({ type: "queryGptOnly", domains: scannedDomains }, (res) => {
        if (chrome.runtime.lastError) {
            // Request may still be processing via popupPort
            return;
        }

        // Handle immediate acknowledgment response
        if (res && res.acknowledging) {
            return;
        }

        // Handle complete response (legacy path)
        if (!res || !res.done) {
            setStatus("Request failed.", "error");
            return;
        }

        // GPT results come in via incremental updates (gptBatchComplete messages)
        if (res.gptResults) {
            gptData = res.gptResults;
        }

        setStatus("Analysis completed!", "success");
        saveState();
    });
}


// ============================================
// Query WHOIS + Analyze (Full Query)
// ============================================
function queryFull() {
    if (scannedDomains.length === 0) {
        setStatus("No domains to query.", "idle");
        return;
    }

    setStatus("Starting WHOIS + Analysis…", "scanning");

    // Establish connection for real-time updates
    connectToProcessor();

    chrome.runtime.sendMessage({ type: "queryWhoisFull", domains: scannedDomains }, (res) => {
        if (chrome.runtime.lastError) {
            // Request may still be processing via popupPort
            return;
        }

        // Handle immediate acknowledgment response
        if (res && res.acknowledging) {
            return;
        }

        // Handle complete response (legacy path)
        if (!res || !res.done) {
            setStatus("Request failed.", "error");
            return;
        }

        whoisData = res.results || {};
        // GPT results might come in via incremental updates, but also final
        if (res.gptResults) {
            gptData = res.gptResults;
        }

        setStatus("WHOIS + Analysis completed!", "success");
        saveState();
    });
}


// ============================================
// Export CSV (Enhanced with SimilarWeb Data)
// ============================================
async function exportCSV() {
    if (scannedDomains.length === 0) {
        setStatus("No domains to export.", "idle");
        return;
    }

    // Confirm whether to include SimilarWeb data
    const includeSimilarWeb = confirm(
        `Export ${scannedDomains.length} domains?\n\n` +
        "Include SimilarWeb data (traffic, rankings, keywords)?\n\n" +
        "✓ Yes - Include (will fetch from cache/API if needed)\n" +
        "✗ No - Skip SimilarWeb columns"
    );

    // Step 1: Batch fetch SimilarWeb data if user wants it
    setStatus("Preparing export data...", "scanning");
    if (includeSimilarWeb) {
        await fetchAllSimilarWeb(scannedDomains);
    }

    // Step 2: Generate CSV with appropriate structure
    let csv;

    const introHeader = currentLanguage === "zh" ? "intro_zh" : "intro";

    if (includeSimilarWeb) {
        // Full CSV with SimilarWeb data (41 columns)
        csv = `domain,category,${introHeader},created_date,age,` +
              "sw_global_rank,sw_country_rank,sw_country_code," +
              "sw_visits,sw_duration_seconds,sw_pages_per_visit,sw_bounce_rate," +
              "sw_visits_month1,sw_visits_month2,sw_visits_month3," +
              "sw_month1_date,sw_month2_date,sw_month3_date," +
              "sw_traffic_search,sw_traffic_direct,sw_traffic_referrals," +
              "sw_traffic_social,sw_traffic_mail,sw_traffic_paid," +
              "sw_top_region1,sw_top_region1_pct,sw_top_region2,sw_top_region2_pct," +
              "sw_top_region3,sw_top_region3_pct," +
              "sw_keyword1,sw_keyword1_traffic,sw_keyword1_volume,sw_keyword1_cpc," +
              "sw_keyword2,sw_keyword2_traffic,sw_keyword2_volume,sw_keyword2_cpc," +
              "sw_keyword3,sw_keyword3_traffic,sw_keyword3_volume,sw_keyword3_cpc\n";
    } else {
        // Lite CSV without SimilarWeb data (5 columns only)
        csv = `domain,category,${introHeader},created_date,age\n`;
    }

    scannedDomains.forEach(domain => {
        // Existing WHOIS & GPT fields
        const item = whoisData[domain];
        const gpt = gptData[domain] || {};
        const cd = item?.result?.domain?.created_date || "";
        const age = calculateAge(cd);

        const introField = currentLanguage === "zh" ? gpt.intro_zh : (gpt.intro || gpt.intro_zh);
        const safeIntro = (introField || "").replace(/,/g, ' ').replace(/\n/g, ' ');
        const safeCat = (gpt.category || "").replace(/,/g, ' ');
        const safeAge = age.replace(/,/g, ' ');

        if (includeSimilarWeb) {
            // SimilarWeb fields (36 columns)
            const swFields = generateSimilarWebCSVFields(domain);
            csv += `${domain},${safeCat},${safeIntro},${cd},${safeAge},${swFields}\n`;
        } else {
            // Lite export without SimilarWeb
            csv += `${domain},${safeCat},${safeIntro},${cd},${safeAge}\n`;
        }
    });

    // Step 3: Download CSV file
    const blob = new Blob([csv], { type: "text/csv;charset=utf-8;" });
    const url = URL.createObjectURL(blob);
    const timestamp = new Date().toISOString().slice(0, 19).replace(/:/g, '-');
    const filename = includeSimilarWeb
        ? `domain_export_full_${timestamp}.csv`
        : `domain_export_lite_${timestamp}.csv`;

    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    setTimeout(() => {
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    }, 100);

    const exportType = includeSimilarWeb ? "full (with SimilarWeb)" : "lite";
    setStatus(`Export completed! (${exportType})`, "success");
}

// ============================================
// Clear List
// ============================================
// ============================================
// Clear List
// ============================================
function deleteRow(domain) {
    // 1. Remove from local list
    scannedDomains = scannedDomains.filter(d => d !== domain);
    delete whoisData[domain];
    delete gptData[domain];

    // 2. Clear from global cache
    chrome.runtime.sendMessage({
        type: "clearCache",
        domains: [domain]
    }, (res) => {
        if (chrome.runtime.lastError) {
            // Ignore - cache clear is best effort
        }
    });

    // 3. Update UI
    saveState();
    renderList();
    setStatus(`Deleted ${domain}`, "success");
}

function deleteHistoryItem(domain) {
    // 1. Clear globally (reuses logic but we need to manage history array manually)
    chrome.runtime.sendMessage({
        type: "clearCache",
        domains: [domain]
    }, (res) => {
        if (chrome.runtime.lastError) {
            // Ignore - cache clear is best effort
        }
    });

    // 2. Remove from global store (whoisData/gptData might have it locally if loaded)
    delete whoisData[domain];
    delete gptData[domain];

    // 3. Remove from scanned list if present
    scannedDomains = scannedDomains.filter(d => d !== domain);

    // 4. Remove from History Arrays
    historyData = historyData.filter(i => i.domain !== domain);
    filteredHistory = filteredHistory.filter(i => i.domain !== domain);

    // 5. Save & Render
    saveState();
    // We don't necessarily need to renderList() effectively unless we switch back,
    // but saveState() persists the scannedDomains removal.

    renderHistoryTable();
}

function clearList() {
    if (scannedDomains.length > 0) {
        // Send message to clear global cache for these domains
        chrome.runtime.sendMessage({
            type: "clearCache",
            domains: scannedDomains
        });
    }

    scannedDomains = [];
    whoisData = {};
    gptData = {};

    setStatus(t("readyToScan"));
    if (countBadge) countBadge.textContent = "0";

    saveState();
    renderList();
}

// ============================================
// Event Listeners (For SimilarWeb fetch only)
// ============================================
// Note: WHOIS and GPT batch updates now use chrome.runtime.connect() for reliable delivery
chrome.runtime.onMessage.addListener((msg) => {
    // This listener handles other messages (like SimilarWeb fetches)
    // Batch updates are now handled through the processorPort connection
});

// ============================================
// Tabs & History State
// ============================================
let currentTab = "scan"; // 'scan' or 'history'
let historyData = []; // Array of { domain, created, risk, ... }
let filteredHistory = [];
let histPage = 1;
const ITEMS_PER_PAGE = 50;

// History Sorting State
let historySortKey = "created"; // default sort by created date
let historySortDir = "desc"; // newest first by default

// ============================================
// Event Listeners (Enhanced)
// ============================================
document.addEventListener("DOMContentLoaded", () => {
    statusBox = document.getElementById("statusBox");
    statusText = document.getElementById("statusText");
    countBadge = document.getElementById("domainCount");

    // Trigger settings sync when popup opens
    chrome.runtime.sendMessage({ type: "syncSettings", force: false }, (response) => {
        if (response && response.success) {
            console.log("[Popup] Settings synced successfully", {
                version: response.settings?.version,
                lastUpdated: response.settings?.lastUpdated
            });
        } else {
            console.warn("[Popup] Settings sync failed:", response?.error);
        }
    });

    // V1.0 SIMPLIFIED: Only bind Scan button
    const btnScan = document.getElementById("btnScan");
    if (btnScan) btnScan.onclick = scanDomains;

    // Bind Feedback button - show contact email
    const btnFeedback = document.getElementById("btnFeedback");
    if (btnFeedback) {
        btnFeedback.addEventListener("click", (e) => {
            e.preventDefault();
            e.stopPropagation();

            const email = "support@websitetrafficchecker.net";
            const message = `Thank you for using WHOIS Checker PRO!\n\nTo send feedback, please contact:\n\n📧 ${email}\n\nThis email address has been copied to your clipboard.`;

            // Show alert with email
            alert(message);

            // Copy email to clipboard
            navigator.clipboard.writeText(email).then(() => {
                console.log("Email copied to clipboard:", email);
            }).catch((err) => {
                console.error("Failed to copy email:", err);
            });
        });
    }

    // Add sort header click handlers
    const sortHeaders = document.querySelectorAll("#domainTable th.sortable");
    sortHeaders.forEach(th => {
        th.addEventListener("click", () => {
            handleSort(th.dataset.key);
        });
    });

    // Bind language toggle
    const langToggle = document.getElementById("langToggle");
    if (langToggle) {
        langToggle.addEventListener("change", (e) => {
            setLanguage(e.target.value);
        });
    }

    // Bind copy all domains button
    const btnCopyAllDomains = document.getElementById("btnCopyAllDomains");
    if (btnCopyAllDomains) {
        btnCopyAllDomains.addEventListener("click", (e) => {
            e.stopPropagation(); // Prevent triggering sort
            copyAllDomains();
        });
    }

    // Initialize language (this will call loadState internally)
    initLanguage();
});

// ============================================
// Tab Logic
// ============================================
function switchTab(tab) {
    currentTab = tab;

    // UI Updates
    document.querySelectorAll(".tab-btn").forEach(b => b.classList.remove("active"));
    document.querySelectorAll(".tab-content").forEach(c => c.classList.remove("active"));

    document.getElementById(`tab${tab.charAt(0).toUpperCase() + tab.slice(1)}`).classList.add("active");
    document.getElementById(`view${tab.charAt(0).toUpperCase() + tab.slice(1)}`).classList.add("active");

    if (tab === "history") {
        loadHistory();
    }
}

// ============================================
// Search Logic (Current Scan)
// ============================================
function getFilteredScanDomains() {
    const searchInput = document.getElementById("scanSearch");
    const query = searchInput ? searchInput.value.toLowerCase().trim() : "";

    if (!query) return scannedDomains;

    return scannedDomains.filter(d => d.toLowerCase().includes(query));
}

// Modify getSortedDomains to use filtered list
function getSortedDomains() {
    // START CHANGE: Use filtered list instead of all scannedDomains
    const domains = getFilteredScanDomains();
    // END CHANGE

    if (currentSortKey === "none") return domains;

    return [...domains].sort((a, b) => {
        // ... (existing sort logic, using 'domains' instead of scannedDomains)
        const itemA = whoisData[a] || {};
        const gptA = gptData[a] || {};
        const itemB = whoisData[b] || {};
        const gptB = gptData[b] || {};

        let valA, valB;

        switch (currentSortKey) {
            case "domain":
                valA = a.toLowerCase();
                valB = b.toLowerCase();
                break;
            case "category":
                valA = (gptA.category || "").toLowerCase();
                valB = (gptB.category || "").toLowerCase();
                break;
            case "intro":
                valA = (gptA.intro_zh || "");
                valB = (gptB.intro_zh || "");
                break;
            case "created":
            case "age":
                const dateA = itemA.result?.domain?.created_date;
                const dateB = itemB.result?.domain?.created_date;
                valA = dateA ? new Date(dateA).getTime() : 0;
                valB = dateB ? new Date(dateB).getTime() : 0;
                break;
        }

        if (valA < valB) return currentSortDir === "asc" ? -1 : 1;
        if (valA > valB) return currentSortDir === "asc" ? 1 : -1;
        return 0;
    });
}

// ============================================
// History Sort Logic
// ============================================
function handleHistorySort(key) {
    if (historySortKey === key) {
        // Toggle direction
        historySortDir = historySortDir === "asc" ? "desc" : "asc";
    } else {
        // New key
        historySortKey = key;
        historySortDir = "asc";
    }
    updateHistorySortUI();
    renderHistoryTable();
}

function updateHistorySortUI() {
    const ths = document.querySelectorAll("#histTable th.sortable");
    ths.forEach(th => {
        th.classList.remove("asc", "desc");
        if (th.dataset.key === historySortKey) {
            th.classList.add(historySortDir);
        }
    });
}

function getSortedHistory() {
    if (!historySortKey) return filteredHistory;

    return [...filteredHistory].sort((a, b) => {
        let valA, valB;

        switch (historySortKey) {
            case "domain":
                valA = a.domain.toLowerCase();
                valB = b.domain.toLowerCase();
                break;
            case "category":
                valA = (a.category || "").toLowerCase();
                valB = (b.category || "").toLowerCase();
                break;
            case "intro":
                valA = (currentLanguage === "zh" ? a.intro_zh : (a.intro || a.intro_zh)) || "";
                valB = (currentLanguage === "zh" ? b.intro_zh : (b.intro || b.intro_zh)) || "";
                break;
            case "created":
            case "age":
                valA = a.created ? new Date(a.created).getTime() : 0;
                valB = b.created ? new Date(b.created).getTime() : 0;
                break;
        }

        if (valA < valB) return historySortDir === "asc" ? -1 : 1;
        if (valA > valB) return historySortDir === "asc" ? 1 : -1;
        return 0;
    });
}

// ============================================
// History Logic
// ============================================
function loadHistory() {
    const histEmpty = document.getElementById("histEmpty");
    histEmpty.style.display = "block";
    histEmpty.textContent = "Loading history...";

    chrome.storage.local.get(["whoisCache", "gptCache"], (res) => {
        const whoisCache = res.whoisCache || {};
        const gptCache = res.gptCache || {};

        // Transform to array
        historyData = Object.keys(whoisCache).map(domain => {
            const item = whoisCache[domain];
            const created = item?.result?.domain?.created_date?.slice(0, 10) || "";
            const gptInfo = gptCache[domain] || {};
            return {
                domain,
                created,
                age: calculateAge(created),
                risk: getRiskLevel(created),
                category: gptInfo.category || "-",
                intro: gptInfo.intro || "-",
                intro_zh: gptInfo.intro_zh || "-"
            };
        });

        // Default sort: Newest first
        historyData.sort((a, b) => {
            const tA = a.created ? new Date(a.created).getTime() : 0;
            const tB = b.created ? new Date(b.created).getTime() : 0;
            return tB - tA;
        });

        filterHistory(); // Initial render
    });
}

function filterHistory() {
    const query = document.getElementById("histSearch").value.toLowerCase().trim();
    const startDate = document.getElementById("histDateStart").value;
    const endDate = document.getElementById("histDateEnd").value;

    filteredHistory = historyData.filter(item => {
        // Domain match
        if (query && !item.domain.toLowerCase().includes(query)) return false;

        // Date Range
        if (startDate && item.created < startDate) return false;
        if (endDate && item.created > endDate) return false;

        return true;
    });

    histPage = 1;
    renderHistoryTable();
}

function changeHistoryPage(delta) {
    const maxPage = Math.ceil(filteredHistory.length / ITEMS_PER_PAGE) || 1;
    let newPage = histPage + delta;
    if (newPage < 1) newPage = 1;
    if (newPage > maxPage) newPage = maxPage;

    histPage = newPage;
    renderHistoryTable();
}

function renderHistoryTable() {
    const tbody = document.getElementById("histBody");
    const empty = document.getElementById("histEmpty");
    const btnPrev = document.getElementById("btnPrev");
    const btnNext = document.getElementById("btnNext");
    const pageInfo = document.getElementById("pageInfo");

    tbody.innerHTML = "";

    if (filteredHistory.length === 0) {
        empty.style.display = "block";
        empty.textContent = "No history found.";
        pageInfo.textContent = "Page 0 / 0";
        return;
    }
    empty.style.display = "none";

    // Apply sorting
    const sortedHistory = getSortedHistory();

    // Slice for pagination
    const start = (histPage - 1) * ITEMS_PER_PAGE;
    const pageData = sortedHistory.slice(start, start + ITEMS_PER_PAGE);
    const maxPage = Math.ceil(sortedHistory.length / ITEMS_PER_PAGE);

    pageData.forEach((item, index) => {
        const tr = document.createElement("tr");

        // Staggered animation
        tr.style.opacity = "0";
        tr.style.animation = `slideIn 0.3s ease forwards ${index * 0.03}s`;

        // Domain with action icons (same as Current Scan)
        const tdDomain = document.createElement("td");
        tdDomain.className = "col-domain";

        // Domain name container
        const domainWrapper = document.createElement("div");
        domainWrapper.className = "domain-wrapper";

        const dSpan = document.createElement("span");
        let riskClass = "domain-low";
        if (item.risk === "high") riskClass = "domain-high";
        if (item.risk === "medium") riskClass = "domain-medium";

        dSpan.className = `domain-name ${riskClass}`;
        dSpan.textContent = item.domain;
        dSpan.title = "Click to copy";
        dSpan.style.cursor = "pointer";
        dSpan.onclick = (e) => {
            e.stopPropagation();
            copyToClipboard(item.domain);
        };
        domainWrapper.appendChild(dSpan);

        // Action icons container
        const actionsDiv = document.createElement("div");
        actionsDiv.className = "domain-actions";

        // 1. Open website
        const openBtn = document.createElement("span");
        openBtn.className = "action-icon";
        openBtn.innerHTML = "🌐";
        openBtn.title = "Open website";
        openBtn.onclick = (e) => {
            e.stopPropagation();
            chrome.tabs.create({ url: `https://${item.domain}` });
        };
        actionsDiv.appendChild(openBtn);

        // 2. Google site: search
        const googleBtn = document.createElement("span");
        googleBtn.className = "action-icon";
        googleBtn.innerHTML = "🔍";
        googleBtn.title = "Google site: search";
        googleBtn.onclick = (e) => {
            e.stopPropagation();
            chrome.tabs.create({ url: `https://www.google.com/search?q=site:${item.domain}` });
        };
        actionsDiv.appendChild(googleBtn);

        // 3. SimilarWeb analysis - always fetch fresh
        const simwebBtn = document.createElement("span");
        simwebBtn.className = "action-icon";
        simwebBtn.innerHTML = "📊";
        simwebBtn.title = "SimilarWeb analysis (fetch from server)";

        simwebBtn.onclick = (e) => {
            e.stopPropagation();
            openSimilarWebModal(item.domain);
        };
        actionsDiv.appendChild(simwebBtn);

        // 4. traffic.cv SEO info
        const seoBtn = document.createElement("span");
        seoBtn.className = "action-icon";
        seoBtn.innerHTML = "📈";
        seoBtn.title = "SEO Info (traffic.cv)";
        seoBtn.onclick = (e) => {
            e.stopPropagation();
            chrome.tabs.create({ url: `https://traffic.cv/${item.domain}` });
        };
        actionsDiv.appendChild(seoBtn);

        // 5. Delete (Trash)
        const delBtn = document.createElement("span");
        delBtn.className = "action-icon";
        delBtn.innerHTML = "🗑️";
        delBtn.title = "Delete & Clear Cache";
        delBtn.style.color = "#ef4444"; // Red
        delBtn.onclick = (e) => {
            e.stopPropagation();
            if (confirm(`Delete ${item.domain} and clear its cache?`)) {
                deleteHistoryItem(item.domain);
            }
        };
        actionsDiv.appendChild(delBtn);

        domainWrapper.appendChild(actionsDiv);
        tdDomain.appendChild(domainWrapper);
        tr.appendChild(tdDomain);

        // Category
        const tdCat = document.createElement("td");
        tdCat.className = "col-cat";
        if (item.category && item.category !== "-") {
            const catBadge = document.createElement("span");
            catBadge.className = "category-badge";
            catBadge.textContent = item.category;
            tdCat.appendChild(catBadge);
        } else {
            tdCat.innerHTML = `<span style="color:#ccc;">-</span>`;
        }
        tr.appendChild(tdCat);

        // Intro (language-aware with fallback)
        const tdIntro = document.createElement("td");
        tdIntro.className = "col-intro";
        const introText = currentLanguage === "zh" ? item.intro_zh : (item.intro || item.intro_zh);
        if (introText && introText !== "-") {
            const div = document.createElement("div");
            div.className = "domain-intro";
            div.textContent = introText;
            div.dataset.tooltip = introText;
            tdIntro.appendChild(div);
        } else {
            tdIntro.innerHTML = `<span style="color:#ccc;">-</span>`;
        }
        tr.appendChild(tdIntro);

        // Date
        const tdDate = document.createElement("td");
        tdDate.className = "col-date";
        tdDate.textContent = item.created || "-";
        tdDate.style.color = "var(--text-muted)";
        tr.appendChild(tdDate);

        // Age
        const tdAge = document.createElement("td");
        tdAge.className = "col-age";
        tdAge.style.textAlign = "right";
        if (item.created) {
            const badge = document.createElement("span");
            badge.className = `age-badge risk-${item.risk}`;
            badge.textContent = item.age;
            tdAge.appendChild(badge);
        } else {
            tdAge.textContent = "-";
        }
        tr.appendChild(tdAge);

        tbody.appendChild(tr);
    });

    // Update Pagination UI
    const totalFormatted = sortedHistory.length >= 1000 ? formatNumber(sortedHistory.length) : sortedHistory.length;
    pageInfo.textContent = `Page ${histPage} / ${maxPage} (${totalFormatted} total)`;
    btnPrev.disabled = histPage === 1;
    btnNext.disabled = histPage === maxPage;

    // Update sort UI indicators
    updateHistorySortUI();
}

async function exportHistoryCSV() {
    if (filteredHistory.length === 0) return;

    // Confirm whether to include SimilarWeb data
    const includeSimilarWeb = confirm(
        `Export ${filteredHistory.length} domains from history?\n\n` +
        "Include SimilarWeb data (traffic, rankings, keywords)?\n\n" +
        "✓ Yes - Include (will fetch from cache/API if needed)\n" +
        "✗ No - Skip SimilarWeb columns"
    );

    // Step 1: Batch fetch SimilarWeb data if user wants it
    const historyDomains = filteredHistory.map(item => item.domain);
    setStatus("Preparing history export...", "scanning");
    if (includeSimilarWeb) {
        await fetchAllSimilarWeb(historyDomains);
    }

    // Step 2: Generate CSV with appropriate structure
    let csv;

    const introHeader = currentLanguage === "zh" ? "intro_zh" : "intro";

    if (includeSimilarWeb) {
        // Full CSV with SimilarWeb data (41 columns)
        csv = `domain,category,${introHeader},created_date,age,` +
              "sw_global_rank,sw_country_rank,sw_country_code," +
              "sw_visits,sw_duration_seconds,sw_pages_per_visit,sw_bounce_rate," +
              "sw_visits_month1,sw_visits_month2,sw_visits_month3," +
              "sw_month1_date,sw_month2_date,sw_month3_date," +
              "sw_traffic_search,sw_traffic_direct,sw_traffic_referrals," +
              "sw_traffic_social,sw_traffic_mail,sw_traffic_paid," +
              "sw_top_region1,sw_top_region1_pct,sw_top_region2,sw_top_region2_pct," +
              "sw_top_region3,sw_top_region3_pct," +
              "sw_keyword1,sw_keyword1_traffic,sw_keyword1_volume,sw_keyword1_cpc," +
              "sw_keyword2,sw_keyword2_traffic,sw_keyword2_volume,sw_keyword2_cpc," +
              "sw_keyword3,sw_keyword3_traffic,sw_keyword3_volume,sw_keyword3_cpc\n";
    } else {
        // Lite CSV without SimilarWeb data (5 columns only)
        csv = `domain,category,${introHeader},created_date,age\n`;
    }

    filteredHistory.forEach(item => {
        const introField = currentLanguage === "zh" ? item.intro_zh : (item.intro || item.intro_zh);
        const safeIntro = (introField || "").replace(/,/g, ' ').replace(/\n/g, ' ');
        const safeCat = (item.category || "").replace(/,/g, ' ');
        const safeAge = (item.age || "").replace(/,/g, ' ');

        if (includeSimilarWeb) {
            // SimilarWeb fields (36 columns)
            const swFields = generateSimilarWebCSVFields(item.domain);
            csv += `${item.domain},${safeCat},${safeIntro},${item.created},${safeAge},${swFields}\n`;
        } else {
            // Lite export without SimilarWeb
            csv += `${item.domain},${safeCat},${safeIntro},${item.created},${safeAge}\n`;
        }
    });

    // Step 3: Download CSV file
    const blob = new Blob([csv], { type: "text/csv;charset=utf-8;" });
    const url = URL.createObjectURL(blob);
    const timestamp = new Date().toISOString().slice(0, 10);
    const filename = includeSimilarWeb
        ? `whois_history_export_full_${timestamp}.csv`
        : `whois_history_export_lite_${timestamp}.csv`;

    // Download logic
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    setTimeout(() => {
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    }, 100);

    const exportType = includeSimilarWeb ? "full (with SimilarWeb)" : "lite";
    setStatus(`History export completed! (${exportType})`, "success");
}

function clearHistoryWhois() {
    if (historyData.length === 0) return;

    if (!confirm("Are you sure you want to clear ALL WHOIS data? This will keep GPT analysis results.")) {
        return;
    }

    const allDomains = historyData.map(d => d.domain);

    // 1. Clear WHOIS from global cache
    chrome.runtime.sendMessage({
        type: "clearWhoisCache",
        domains: allDomains
    }, (res) => {
        if (chrome.runtime.lastError) {
            // Ignore - cache clear is best effort
        }
    });

    // 2. Clear WHOIS from local store
    allDomains.forEach(d => {
        delete whoisData[d]; // Delete local WHOIS
        // Keep gptData[d]
    });

    // 3. Update History Array (Reset WHOIS fields)
    historyData.forEach(item => {
        item.created = "";
        item.age = "";
        item.risk = "low";
    });

    // 4. Save & Render
    saveState();
    renderHistoryTable();
    setStatus("Cleared WHOIS cache for all history.", "success");
}

function resetHistoryItemWhois(domain) {
    // 1. Clear WHOIS from global cache
    chrome.runtime.sendMessage({
        type: "clearWhoisCache",
        domains: [domain]
    });

    // 2. Clear local WHOIS
    delete whoisData[domain];

    // 3. Update History Array (Reset WHOIS fields)
    const item = historyData.find(i => i.domain === domain);
    if (item) {
        item.created = "";
        item.age = "";
        item.risk = "low";
    }

    // 4. Update UI
    renderHistoryTable();
    setStatus(`Reset WHOIS for ${domain}`, "success");
}

// ============================================
// SimilarWeb Modal Logic
// ============================================
// ============================================
// SimilarWeb Modal Logic
// ============================================
async function openSimilarWebModal(domain) {
    try {
        const modal = document.getElementById("simWebModal");
        if (!modal) {
            console.error("Modal element #simWebModal not found in DOM.");
            return;
        }

        const loader = document.getElementById("swLoader");
        const content = document.getElementById("swContent");

        // 1. Reset & Open Modal immediately
        const elDomain = document.getElementById("swDomain");
        if (elDomain) elDomain.textContent = domain;

        const elDesc = document.getElementById("swDesc");
        if (elDesc) elDesc.textContent = "Loading analysis...";

        const elThumb = document.getElementById("swScreenshot");
        if (elThumb) elThumb.style.display = "none";

        // Show Loader if available
        if (loader) loader.style.display = "flex";
        if (content) content.style.display = "none";

        modal.style.display = "flex";

        // Close Handler with refresh
        const closeModal = () => {
            modal.style.display = "none";
            // Refresh list when modal closes to update icon status
            if (currentTab === "scan") {
                renderList();
            } else if (currentTab === "history") {
                renderHistoryTable();
            }
        };

        const btnClose = document.getElementById("btnCloseModal");
        if (btnClose) {
            btnClose.onclick = closeModal;
        }
        modal.onclick = (e) => {
            if (e.target === modal) closeModal();
        };

        // 2. Fetch data (Async)
        console.log("Requesting SimilarWeb data for:", domain);
        const res = await new Promise(resolve => {
            chrome.runtime.sendMessage({ type: "fetchSimilarWeb", domain }, (response) => {
                if (chrome.runtime.lastError) {
                    console.error("SimilarWeb fetch error:", chrome.runtime.lastError);
                    resolve(null);
                } else {
                    resolve(response);
                }
            });
        });

        // 3. Handle Response
        console.log("SimilarWeb API response:", res);
        console.log("Response structure check:", {
            hasCode: !!res.code,
            hasData: !!res.data,
            hasDataData: !!(res.data && res.data.data),
            hasSuccess: !!res.success
        });

        // Handle both response formats:
        // Backend format: { code: 0, data: { domain, data: {...} } }
        // Extension format: { success: true, data: {...} }
        let similarWebData = null;

        if (res.code === 0 && res.data && res.data.data) {
            // Backend format - convert snake_case to PascalCase for frontend
            const backendData = res.data.data;
            console.log("Backend data before transform:", backendData);
            similarWebData = transformBackendToFrontend(backendData);
            console.log("Transformed data:", similarWebData);
        } else if (res.success && res.data) {
            // Extension format (already in correct format)
            similarWebData = res.data;
            console.log("Using extension format data:", similarWebData);
        }

        if (!similarWebData) {
            if (loader) loader.style.display = "none";
            if (elDesc) elDesc.textContent = "Data unavailable or request failed.";
            console.error("Invalid response format:", res);
            return;
        }

        console.log("Final similarWebData to render:", similarWebData);

        // 4. Update local cache with new data
        simWebCache[domain] = {
            timestamp: Date.now(),
            data: similarWebData
        };

        // 5. Render & Switch View
        renderSimilarWebModal(similarWebData);
        if (loader) loader.style.display = "none";
        if (content) content.style.display = "block";

        // 6. Refresh the domain list to update icon status
        // Use setTimeout to avoid blocking the modal display
        setTimeout(() => {
            if (currentTab === "scan") {
                renderList();
            } else if (currentTab === "history") {
                renderHistoryTable();
            }
        }, 100);

    } catch (e) {
        console.error("Error opening SimilarWeb modal:", e);
    }
}

/**
 * Transform backend snake_case format to frontend PascalCase format
 */
function transformBackendToFrontend(backendData) {
    return {
        // Basic info
        Domain: backendData.domain, // Add domain field
        SiteName: backendData.title || backendData.domain,
        Description: backendData.description,
        LargeScreenshot: backendData.screenshot_url,

        // Rankings
        GlobalRank: {
            Rank: backendData.global_rank
        },
        CountryRank: {
            Rank: backendData.country_rank,
            CountryCode: backendData.country_code
        },

        // Engagement metrics
        Engagments: {
            Visits: backendData.monthly_visits,
            TimeOnSite: backendData.avg_visit_duration,
            PagePerVisit: backendData.pages_per_visit,
            BounceRate: backendData.bounce_rate
        },

        // Traffic sources (convert to PascalCase keys)
        TrafficSources: backendData.traffic_sources ? {
            Search: backendData.traffic_sources.organic,
            Direct: backendData.traffic_sources.direct,
            Referrals: backendData.traffic_sources.referral,
            Social: backendData.traffic_sources.social,
            Mail: backendData.traffic_sources.mail,
            "Paid Referrals": backendData.traffic_sources.paid
        } : {},

        // Top countries
        TopCountryShares: (backendData.top_countries || []).map(c => ({
            CountryCode: c.country_code,
            Country: c.country,
            Value: c.share
        })),

        // Top keywords
        TopKeywords: (backendData.top_keywords || []).map(k => ({
            Name: k.name,
            EstimatedValue: k.estimated_value,
            Volume: k.volume,
            Cpc: k.cpc
        })),

        // Monthly visits
        EstimatedMonthlyVisits: backendData.estimated_monthly_visits || {}
    };
}

function renderSimilarWebModal(data) {
    const modal = document.getElementById("simWebModal");
    if (!modal) return;

    // Header - Show domain name, not title
    // Extract domain from data (could be in different fields)
    let domainName = "";
    if (data.Domain) {
        domainName = data.Domain;
    } else if (data.SiteName && !data.SiteName.includes(' ')) {
        // If SiteName looks like a domain (no spaces), use it
        domainName = data.SiteName;
    } else {
        // Try to extract from other sources
        domainName = data.SiteName || "Unknown";
    }

    document.getElementById("swDomain").textContent = domainName;
    document.getElementById("swDesc").textContent = data.Description || "No description available.";

    // Domain Link
    const domainLink = document.getElementById("swDomainLink");
    if (domainLink && domainName !== "Unknown") {
        domainLink.href = `https://${domainName}`;
    }

    // Screenshot
    const thumb = document.getElementById("swScreenshot");
    if (data.LargeScreenshot) {
        thumb.src = data.LargeScreenshot;
        thumb.style.display = "block";
    } else {
        thumb.style.display = "none";
    }

    // Ranks (Format with K/M/B if large numbers)
    document.getElementById("swGlobal").textContent = data.GlobalRank?.Rank ? `#${formatNumber(data.GlobalRank.Rank)}` : "-";
    document.getElementById("swCountry").textContent = data.CountryRank?.Rank ? `#${formatNumber(data.CountryRank.Rank)} (${data.CountryRank.CountryCode})` : "-";

    // Stats
    const eng = data.Engagments || {};
    document.getElementById("swVisits").textContent = eng.Visits ? formatNumber(parseInt(eng.Visits)) : "-";

    // Format Duration
    const seconds = parseInt(eng.TimeOnSite || 0);
    const mm = Math.floor(seconds / 60).toString().padStart(2, '0');
    const ss = (seconds % 60).toString().padStart(2, '0');
    document.getElementById("swDuration").textContent = `${mm}:${ss}`;

    document.getElementById("swPages").textContent = eng.PagePerVisit ? parseFloat(eng.PagePerVisit).toFixed(2) : "-";
    document.getElementById("swBounce").textContent = eng.BounceRate ? `${(parseFloat(eng.BounceRate) * 100).toFixed(2)}%` : "-";

    // --- Charts ---

    // 1. Visits Line Chart (CSS Bars for simplicity)
    // 1. Visits Chart (SVG Area Chart)
    const visitsContainer = document.getElementById("visitsChart");
    visitsContainer.innerHTML = "";

    // Sort keys (dates)
    const keys = Object.keys(data.EstimatedMonthlyVisits || {}).sort();
    const values = keys.map(k => data.EstimatedMonthlyVisits[k]);

    if (keys.length > 0) {
        // Dimensions
        const width = visitsContainer.clientWidth || 380;
        const height = 150; // SVG height
        const padding = { top: 20, right: 10, bottom: 30, left: 40 };
        const chartW = width - padding.left - padding.right;
        const chartH = height - padding.top - padding.bottom;

        // Find Max for Y Scale
        const maxVal = Math.max(...values, 0);
        // Round up max for nicer grid (e.g. 4.2M -> 5M)
        let yMax = maxVal * 1.1;
        if (yMax === 0) yMax = 100;

        // Helper: Get Coordinates
        const getX = (i) => padding.left + (i / (keys.length - 1)) * chartW;
        const getY = (val) => padding.top + chartH - (val / yMax * chartH);

        // Generate Paths
        let pathD = "";
        let areaD = "";

        values.forEach((val, i) => {
            const x = getX(i);
            const y = getY(val);
            if (i === 0) {
                pathD += `M ${x} ${y}`;
                areaD += `M ${x} ${y}`;
            } else {
                pathD += ` L ${x} ${y}`;
                areaD += ` L ${x} ${y}`;
            }
        });

        // Close area path
        areaD += ` L ${getX(values.length - 1)} ${padding.top + chartH} L ${padding.left} ${padding.top + chartH} Z`;

        // Generate SVG
        const svgNs = "http://www.w3.org/2000/svg";
        const svg = document.createElementNS(svgNs, "svg");
        svg.setAttribute("width", width);
        svg.setAttribute("height", height);

        // Defs (Gradient)
        const defs = document.createElementNS(svgNs, "defs");
        const grad = document.createElementNS(svgNs, "linearGradient");
        grad.id = "areaGradient";
        grad.setAttribute("x1", "0");
        grad.setAttribute("x2", "0");
        grad.setAttribute("y1", "0");
        grad.setAttribute("y2", "1");

        const stop1 = document.createElementNS(svgNs, "stop");
        stop1.setAttribute("offset", "0%");
        stop1.setAttribute("stop-color", "#3b82f6");
        stop1.setAttribute("stop-opacity", "0.3");

        const stop2 = document.createElementNS(svgNs, "stop");
        stop2.setAttribute("offset", "100%");
        stop2.setAttribute("stop-color", "#3b82f6");
        stop2.setAttribute("stop-opacity", "0");

        grad.appendChild(stop1);
        grad.appendChild(stop2);
        defs.appendChild(grad);
        svg.appendChild(defs);

        // Y-Axis Grid & Labels (3 lines)
        const gridCount = 3;
        for (let i = 0; i <= gridCount; i++) {
            const val = (yMax / gridCount) * i;
            const y = getY(val);

            // Grid Line
            if (i > 0) { // Don't draw bottom line if we use axis
                const line = document.createElementNS(svgNs, "line");
                line.setAttribute("x1", padding.left);
                line.setAttribute("x2", width - padding.right);
                line.setAttribute("y1", y);
                line.setAttribute("y2", y);
                line.setAttribute("stroke", "#e2e8f0");
                line.setAttribute("stroke-dasharray", "4");
                svg.appendChild(line);
            }

            // Label
            const text = document.createElementNS(svgNs, "text");
            text.setAttribute("x", padding.left - 5);
            text.setAttribute("y", y + 4);
            text.setAttribute("text-anchor", "end");
            text.setAttribute("font-size", "10");
            text.setAttribute("fill", "#94a3b8");
            text.textContent = formatNumber(val);
            svg.appendChild(text);
        }

        // Area Path
        const pathArea = document.createElementNS(svgNs, "path");
        pathArea.setAttribute("d", areaD);
        pathArea.setAttribute("fill", "url(#areaGradient)");
        svg.appendChild(pathArea);

        // Line Path
        const pathLine = document.createElementNS(svgNs, "path");
        pathLine.setAttribute("d", pathD);
        pathLine.setAttribute("fill", "none");
        pathLine.setAttribute("stroke", "#3b82f6");
        pathLine.setAttribute("stroke-width", "2");
        svg.appendChild(pathLine);

        // Points (Dots)
        values.forEach((val, i) => {
            const circle = document.createElementNS(svgNs, "circle");
            circle.setAttribute("cx", getX(i));
            circle.setAttribute("cy", getY(val));
            circle.setAttribute("r", "3");
            circle.setAttribute("fill", "#3b82f6");
            circle.setAttribute("class", "chart-dot");
            svg.appendChild(circle);
        });

        // X-Axis Labels (Dates)
        keys.forEach((k, i) => {
            const x = getX(i);
            const text = document.createElementNS(svgNs, "text");
            text.setAttribute("x", x);
            text.setAttribute("y", height - 10);
            text.setAttribute("text-anchor", "middle");
            text.setAttribute("font-size", "10");
            text.setAttribute("fill", "#64748b");
            // k is "2025-09-01", format to "2025/09"
            const label = k.slice(0, 7).replace(/-/g, '/');
            text.textContent = label;
            svg.appendChild(text);
        });

        // Interaction Layer
        const overlay = document.createElementNS(svgNs, "rect");
        overlay.setAttribute("width", width);
        overlay.setAttribute("height", height);
        overlay.setAttribute("fill", "transparent");
        svg.appendChild(overlay);

        // Tooltip Rendering
        let tooltip = document.getElementById("chartTooltip");
        if (!tooltip) {
            tooltip = document.createElement("div");
            tooltip.id = "chartTooltip";
            tooltip.className = "chart-tooltip";
            visitsContainer.appendChild(tooltip); // Append to container, absolute pos
        }

        overlay.onmousemove = (e) => {
            const rect = svg.getBoundingClientRect();
            const mouseX = e.clientX - rect.left - padding.left;

            // Find closest index
            const step = chartW / (keys.length - 1);
            let idx = Math.round(mouseX / step);
            if (idx < 0) idx = 0;
            if (idx > keys.length - 1) idx = keys.length - 1;

            const k = keys[idx];
            const v = values[idx];

            // Show Tooltip
            tooltip.style.display = "block";
            tooltip.style.left = (getX(idx) + 10) + "px"; // Offset slightly
            tooltip.style.top = (getY(v) - 40) + "px";
            tooltip.innerHTML = `<strong>${k.slice(0, 7).replace(/-/g, '/')}</strong><br>Visits: ${formatNumber(v)}`;
        };

        overlay.onmouseleave = () => {
            tooltip.style.display = "none";
        };

        visitsContainer.appendChild(svg);

        // Clean up old date labels if any remain in HTML logic
        const dateContainer = document.querySelector(".chart-dates");
        if (dateContainer) dateContainer.style.display = "none";
    }

    // 2. Traffic Sources Donut (Conic Gradient)
    const sources = data.TrafficSources || {};
    const COLORS = {
        "Search": "#ef4444",   // Red
        "Direct": "#3b82f6",   // Blue
        "Referrals": "#10b981",// Green
        "Social": "#a855f7",   // Purple
        "Mail": "#f97316",     // Orange
        "Paid Referrals": "#84cc16" // Lime
    };

    let gradientStr = "";
    let currentPct = 0;
    const legendBox = document.getElementById("trafficLegend");
    legendBox.innerHTML = "";

    Object.entries(sources).sort((a, b) => b[1] - a[1]).forEach(([key, val]) => {
        const pct = val * 100;
        const color = COLORS[key] || "#94a3b8"; // Default slate

        // Add to gradient
        const endPct = currentPct + pct;
        gradientStr += `${color} ${currentPct}% ${endPct}%, `;
        currentPct = endPct;

        // Add to Legend
        const item = document.createElement("div");
        item.className = "legend-item";
        item.innerHTML = `<div class="legend-color" style="background:${color}"></div> <span>${key}: ${pct.toFixed(2)}%</span>`;
        legendBox.appendChild(item);
    });

    const donut = document.getElementById("trafficDonut");
    if (gradientStr) {
        donut.style.background = `conic-gradient(${gradientStr.slice(0, -2)})`;
    }

    // 3. Top Regions
    const regionBody = document.getElementById("topRegions");
    regionBody.innerHTML = "";
    (data.TopCountryShares || []).forEach(c => {
        const tr = document.createElement("tr");

        // Region Name (Code)
        const tdName = document.createElement("td");
        tdName.textContent = c.CountryCode || "Unknown";
        tr.appendChild(tdName);

        // Percentage
        const tdPct = document.createElement("td");
        tdPct.className = "text-end";
        tdPct.innerHTML = `<span class="kw-pill blue">${(c.Value * 100).toFixed(2)}%</span>`;
        tr.appendChild(tdPct);

        regionBody.appendChild(tr);
    });

    // 4. Top Keywords
    const keywordBody = document.getElementById("topKeywords");
    keywordBody.innerHTML = "";
    (data.TopKeywords || []).slice(0, 5).forEach(k => {
        const tr = document.createElement("tr");

        // Name with Icon
        const tdName = document.createElement("td");

        const wrapDiv = document.createElement("div");
        wrapDiv.className = "kw-cell-wrap";

        // Text
        const textSpan = document.createElement("span");
        textSpan.className = "kw-text";
        textSpan.textContent = k.Name;
        textSpan.title = k.Name; // Full text on hover
        wrapDiv.appendChild(textSpan);

        // Icon
        const linkIcon = document.createElement("span");
        linkIcon.className = "ext-link-icon";
        // Simple square-arrow icon
        linkIcon.innerHTML = `<svg viewBox="0 0 24 24" width="10" height="10" stroke="currentColor" stroke-width="2.5" fill="none" stroke-linecap="round" stroke-linejoin="round"><path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"></path><polyline points="15 3 21 3 21 9"></polyline><line x1="10" y1="14" x2="21" y2="3"></line></svg>`;
        linkIcon.title = "Search on Google";
        linkIcon.onclick = (e) => {
            e.stopPropagation();
            chrome.tabs.create({ url: `https://www.google.com/search?q=${encodeURIComponent(k.Name)}` });
        };
        wrapDiv.appendChild(linkIcon);

        tdName.appendChild(wrapDiv);
        tr.appendChild(tdName);

        // Traffic (EstimatedValue)
        const tdTraffic = document.createElement("td");
        tdTraffic.className = "text-end";
        tdTraffic.innerHTML = `<span class="kw-pill purple">${formatNumber(k.EstimatedValue)}</span>`;
        tr.appendChild(tdTraffic);

        // Volume
        const tdVol = document.createElement("td");
        tdVol.className = "text-end";
        tdVol.innerHTML = `<span class="kw-pill purple">${formatNumber(k.Volume)}</span>`;
        tr.appendChild(tdVol);

        // CPC
        const tdCpc = document.createElement("td");
        tdCpc.className = "text-end";
        tdCpc.innerHTML = `<span class="kw-pill blue">$${(k.Cpc || 0).toFixed(2)}</span>`;
        tr.appendChild(tdCpc);

        keywordBody.appendChild(tr);
    });

    // WHOIS Information
    renderWhoisInfo(data);

    // Show Modal (Failsafe)
    if (modal.style.display !== "flex") modal.style.display = "flex";
}

/**
 * Render WHOIS information in the modal
 */
function renderWhoisInfo(data) {
    // Get WHOIS data using the actual domain name (not the site title)
    const domainName = data.Domain || data.SiteName || "";
    const whoisInfo = whoisData[domainName] || {};
    const domainInfo = whoisInfo.result?.domain || {};

    console.log("renderWhoisInfo - Domain:", domainName, "whoisInfo:", whoisInfo);

    // Domain Creation
    const createdEl = document.getElementById("swCreated");
    if (createdEl) {
        const createdDate = domainInfo.created_date;
        createdEl.textContent = createdDate ? formatDate(createdDate) : "-";
    }

    // Domain Expiration
    const expiresEl = document.getElementById("swExpires");
    if (expiresEl) {
        const expiresDate = domainInfo.expiration_date;
        expiresEl.textContent = expiresDate ? formatDate(expiresDate) : "-";
    }

    // Last Changed
    const updatedEl = document.getElementById("swUpdated");
    if (updatedEl) {
        const updatedDate = domainInfo.updated_date;
        updatedEl.textContent = updatedDate ? formatDate(updatedDate) : "-";
    }

    // Domain Age
    const ageEl = document.getElementById("swAge");
    if (ageEl) {
        const createdDate = domainInfo.created_date;
        if (createdDate) {
            const age = calculateAgeModal(createdDate);
            ageEl.textContent = age || "-";
        } else {
            ageEl.textContent = "-";
        }
    }
}

/**
 * Format date to YYYY-MM-DD
 */
function formatDate(dateStr) {
    if (!dateStr) return "-";
    try {
        const date = new Date(dateStr);
        if (isNaN(date.getTime())) return "-";
        return date.toISOString().split('T')[0];
    } catch (e) {
        return "-";
    }
}

/**
 * Calculate domain age (simplified version for modal)
 * Accepts both string and Date formats
 */
function calculateAgeModal(createdDate) {
    if (!createdDate) return "-";

    // If it's a string, convert to Date
    let dateObj = createdDate;
    if (typeof createdDate === 'string') {
        dateObj = new Date(createdDate);
    }

    if (isNaN(dateObj.getTime())) return "-";

    const now = new Date();
    const diffMs = now - dateObj;
    const diffYears = diffMs / (1000 * 60 * 60 * 24 * 365.25);

    if (diffYears < 0) return "-";
    return `${diffYears.toFixed(2)} years`;
}

// ============================================
// Number Formatting Utility (K/M/B Standard)
// ============================================
/**
 * Format numbers according to design standards:
 * - 1K ~ 999K: Use K (thousands)
 * - 1M ~ 999M: Use M (millions)
 * - 1B+: Use B (billions)
 * - Keep 1 decimal place for precision
 * - Add space between number and unit
 * - Avoid decimals < 1 (e.g., use "500" instead of "0.5K")
 */
function formatNumber(num) {
    if (!num || isNaN(num)) return "-";

    const absNum = Math.abs(num);

    // Billions (≥ 1,000,000,000)
    if (absNum >= 1000000000) {
        const val = num / 1000000000;
        // Round to integer if close to whole number, otherwise 1 decimal
        return val % 1 === 0 ? `${Math.round(val)} B` : `${val.toFixed(1)} B`;
    }

    // Millions (≥ 1,000,000)
    if (absNum >= 1000000) {
        const val = num / 1000000;
        return val % 1 === 0 ? `${Math.round(val)} M` : `${val.toFixed(1)} M`;
    }

    // Thousands (≥ 1,000)
    if (absNum >= 1000) {
        const val = num / 1000;
        return val % 1 === 0 ? `${Math.round(val)} K` : `${val.toFixed(1)} K`;
    }

    // Small numbers (< 1,000): Show as-is with comma separators
    return Math.round(num).toLocaleString();
}

// Legacy alias for backward compatibility
function formatK(num) {
    return formatNumber(num);
}

// ============================================
// SimilarWeb Data Access Helper
// ============================================
/**
 * Retrieve cached SimilarWeb data for a domain
 * @param {string} domain - Domain name
 * @returns {object|null} - Returns cached data if available, null otherwise
 *
 * NOTE: Uses local simWebCache filled by fetchAllSimilarWeb()
 * The backend also implements database caching (24h), so combined approach:
 * - Frontend cache: Quick retrieval for export/display
 * - Backend cache: Database-level caching for API calls
 */
function getSimilarWebData(domain) {
    // Return cached data if available (filled by fetchAllSimilarWeb)
    return simWebCache[domain]?.data || null;
}

// ============================================
// Batch SimilarWeb Fetching
// ============================================
/**
 * Fetch SimilarWeb data for all domains that don't have valid cached data
 * @param {Array<string>} domains - Array of domain names
 * @returns {Promise<void>}
 */
async function fetchAllSimilarWeb(domains) {
    if (!domains || domains.length === 0) return;

    // Always fetch all domains - no local cache check
    const domainsToFetch = domains;

    console.log(`Fetching SimilarWeb data for ${domainsToFetch.length} domains from server...`);

    let completed = 0;
    const total = domainsToFetch.length;

    for (const domain of domainsToFetch) {
        try {
            // Update progress
            completed++;
            setStatus(`Fetching SimilarWeb data... (${completed}/${total})`, "scanning");

            // Fetch data via background script
            const response = await new Promise((resolve) => {
                chrome.runtime.sendMessage(
                    { type: "fetchSimilarWeb", domain },
                    resolve
                );
            });

            // Update local cache if successful
            // Backend returns: { code: 0, data: { domain, source, data: {...}, metadata: {...} } }
            if (response && response.code === 0 && response.data && response.data.data) {
                // Transform snake_case backend data to PascalCase frontend format
                const transformedData = transformBackendToFrontend(response.data.data);
                simWebCache[domain] = {
                    timestamp: Date.now(),
                    data: transformedData
                };
                console.log(`SimilarWeb cache updated for ${domain}, from_cache: ${response.data.metadata?.from_cache}`);
            } else {
                console.warn(`Failed to fetch SimilarWeb data for ${domain}`, response);
            }

            // Rate limiting: Wait 2-3 seconds between requests
            if (completed < total) {
                const delay = 2000 + Math.random() * 1000; // 2-3 seconds
                await new Promise(resolve => setTimeout(resolve, delay));
            }

        } catch (error) {
            console.error(`Error fetching SimilarWeb data for ${domain}:`, error);
            // Continue with next domain
        }
    }

    // Save updated cache to storage for persistence
    try {
        await chrome.storage.local.set({ simWebCache });
        console.log("SimilarWeb cache saved to storage");
    } catch (error) {
        console.error("Failed to save SimilarWeb cache to storage:", error);
    }

    console.log(`SimilarWeb batch fetch completed: ${completed}/${total} domains`);
}

// ============================================
// SimilarWeb CSV Field Generator
// ============================================
/**
 * Generate CSV fields for SimilarWeb data
 * @param {string} domain - Domain name
 * @returns {string} - Comma-separated CSV fields (36 columns)
 */
function generateSimilarWebCSVFields(domain) {
    let swData = getSimilarWebData(domain);

    // If no cached data is available, return empty fields
    // (Data should have been fetched by fetchAllSimilarWeb before calling this)
    if (!swData) {
        return Array(36).fill('').join(',');
    }

    // Transform backend snake_case format to frontend PascalCase if needed
    if (swData.global_rank !== undefined || swData.monthly_visits !== undefined) {
        swData = transformBackendToFrontend(swData);
    }

    const fields = [];

    // Helper: Sanitize CSV value (escape commas, quotes, newlines)
    const sanitize = (val) => {
        if (val === null || val === undefined) return '';
        const str = String(val).replace(/,/g, ' ').replace(/\n/g, ' ').replace(/"/g, '""');
        return str;
    };

    // Helper: Format percentage (0.45 -> "45.00%")
    const formatPct = (val) => {
        if (!val || isNaN(val)) return '';
        return `${(val * 100).toFixed(2)}%`;
    };

    // === Rankings (3 fields) ===
    fields.push(sanitize(swData.GlobalRank?.Rank || ''));
    fields.push(sanitize(swData.CountryRank?.Rank || ''));
    fields.push(sanitize(swData.CountryRank?.CountryCode || ''));

    // === Engagement Metrics (4 fields) ===
    const eng = swData.Engagments || {};
    fields.push(sanitize(eng.Visits || ''));
    fields.push(sanitize(eng.TimeOnSite || ''));
    fields.push(sanitize(eng.PagePerVisit || ''));
    fields.push(formatPct(eng.BounceRate));

    // === Historical Visits (6 fields) - Last 3 months ===
    const monthlyVisits = swData.EstimatedMonthlyVisits || {};
    const sortedMonths = Object.keys(monthlyVisits).sort().reverse(); // Newest first

    for (let i = 0; i < 3; i++) {
        if (sortedMonths[i]) {
            fields.push(sanitize(monthlyVisits[sortedMonths[i]])); // Visit count
        } else {
            fields.push('');
        }
    }

    for (let i = 0; i < 3; i++) {
        if (sortedMonths[i]) {
            // Format date as "2025-11" (YYYY-MM)
            fields.push(sanitize(sortedMonths[i].slice(0, 7)));
        } else {
            fields.push('');
        }
    }

    // === Traffic Sources (6 fields) ===
    const sources = swData.TrafficSources || {};
    fields.push(formatPct(sources.Search));
    fields.push(formatPct(sources.Direct));
    fields.push(formatPct(sources.Referrals));
    fields.push(formatPct(sources.Social));
    fields.push(formatPct(sources.Mail));
    fields.push(formatPct(sources['Paid Referrals']));

    // === Top Regions (6 fields) - Top 3 countries ===
    const regions = swData.TopCountryShares || [];
    for (let i = 0; i < 3; i++) {
        if (regions[i]) {
            fields.push(sanitize(regions[i].CountryCode || ''));
            fields.push(formatPct(regions[i].Value));
        } else {
            fields.push('');
            fields.push('');
        }
    }

    // === Top Keywords (12 fields) - Top 3 keywords ===
    const keywords = swData.TopKeywords || [];
    for (let i = 0; i < 3; i++) {
        if (keywords[i]) {
            fields.push(sanitize(keywords[i].Name || ''));
            fields.push(sanitize(keywords[i].EstimatedValue || ''));
            fields.push(sanitize(keywords[i].Volume || ''));
            fields.push(sanitize(keywords[i].Cpc ? `$${keywords[i].Cpc.toFixed(2)}` : ''));
        } else {
            fields.push('');
            fields.push('');
            fields.push('');
            fields.push('');
        }
    }

    return fields.join(',');
}

// Helper: Copy to Clipboard
function copyToClipboard(text) {
    if (!text) return;
    navigator.clipboard.writeText(text).then(() => {
        const copyMsg = t("copying");
        const msg = typeof copyMsg === "function" ? copyMsg(text) : `Copied "${text}"`;
        setStatus(msg, "success");
    }).catch(err => {
        console.error("Failed to copy:", err);
        setStatus(t("copyFailed"), "error");
    });
}

// Helper: Copy All Domains to Clipboard
function copyAllDomains() {
    if (scannedDomains.length === 0) {
        setStatus("No domains to copy.", "idle");
        return;
    }

    const domainList = scannedDomains.join('\n');
    navigator.clipboard.writeText(domainList).then(() => {
        const count = scannedDomains.length;
        const msg = currentLanguage === "zh"
            ? `已复制 ${count} 个域名`
            : `Copied ${count} domains`;
        setStatus(msg, "success");

        // Visual feedback on button
        const btn = document.getElementById("btnCopyAllDomains");
        if (btn) {
            btn.classList.add("copied");
            setTimeout(() => btn.classList.remove("copied"), 1500);
        }
    }).catch(err => {
        console.error("Failed to copy domains:", err);
        setStatus(t("copyFailed"), "error");
    });
}
