/**
 * Settings Synchronization Module
 *
 * Handles syncing extension settings with backend API
 */

import { CONFIG } from './config.js';

console.log("[settings-sync] Module loaded");

// Settings cache in memory
let settingsCache = {
    domainSettings: null,
    loadTime: null
};

/**
 * Fetch settings from backend API
 * @returns {Promise<Object>} Settings object or null if failed
 */
async function fetchSettingsFromBackend() {
    const url = CONFIG.getBackendUrl(CONFIG.BACKEND.ENDPOINTS.SETTINGS_PROXY);

    try {
        CONFIG.log(`Fetching settings from: ${url}`);

        const response = await fetch(url, {
            method: 'GET',
            headers: CONFIG.EXTENSION.getAuthHeaders()
        });

        if (!response.ok) {
            CONFIG.warn(`Settings API returned ${response.status}`);
            return null;
        }

        const result = await response.json();

        // Backend returns: { code: 0, message: "ok", data: { settings: {...} } }
        if (result.code === 0 && result.data && result.data.settings) {
            CONFIG.log("Settings fetched successfully", {
                version: result.data.settings.version,
                lastUpdated: result.data.settings.lastUpdated,
                suffixCount: result.data.settings.allowedSuffixes?.length || 0
            });

            return result.data.settings;
        }

        CONFIG.warn("Invalid settings response format", result);
        return null;

    } catch (error) {
        CONFIG.error("Failed to fetch settings from backend", error.message);
        return null;
    }
}

/**
 * Save settings to chrome.storage.local
 * @param {Object} settings - Settings object to save
 * @returns {Promise<void>}
 */
async function saveSettingsToStorage(settings) {
    return new Promise((resolve, reject) => {
        chrome.storage.local.set({ domainSettings: settings }, () => {
            if (chrome.runtime.lastError) {
                reject(chrome.runtime.lastError);
            } else {
                resolve();
            }
        });
    });
}

/**
 * Get settings from chrome.storage.local
 * @returns {Promise<Object>} Settings object
 */
async function getSettingsFromStorage() {
    return new Promise((resolve) => {
        chrome.storage.local.get(['domainSettings'], (result) => {
            resolve(result.domainSettings || null);
        });
    });
}

/**
 * Initialize settings from backend on extension startup
 * First tries to load from storage, then syncs with backend in background
 *
 * @returns {Promise<Object>} Settings object (may be from cache)
 */
export async function initializeSettings() {
    console.log("[settings-sync] Initializing settings...");

    try {
        // First, try to load from local storage (fast path)
        const cachedSettings = await getSettingsFromStorage();

        if (cachedSettings && cachedSettings.allowedSuffixes && Array.isArray(cachedSettings.allowedSuffixes)) {
            CONFIG.log("Loaded cached settings from storage", {
                version: cachedSettings.version,
                lastUpdated: cachedSettings.lastUpdated,
                suffixCount: cachedSettings.allowedSuffixes.length
            });

            settingsCache.domainSettings = cachedSettings;
            settingsCache.loadTime = new Date(cachedSettings.lastUpdated);

            // Sync with backend in background (non-blocking)
            syncSettings(false).catch(err => {
                CONFIG.warn("Background sync failed (non-critical)", err.message);
            });

            return cachedSettings;
        }

        // No cached settings, fetch from backend
        CONFIG.log("No cached settings found, fetching from backend");
        const settings = await fetchSettingsFromBackend();

        if (settings) {
            await saveSettingsToStorage(settings);
            settingsCache.domainSettings = settings;
            settingsCache.loadTime = new Date(settings.lastUpdated);
            return settings;
        }

        // Backend also failed, return empty settings
        CONFIG.warn("Backend fetch failed, returning empty settings");
        return {};

    } catch (error) {
        CONFIG.error("Failed to initialize settings", error);
        return {};
    }
}

/**
 * Sync settings with backend API
 *
 * @param {boolean} force - Force sync even if recently synced
 * @returns {Promise<Object>} Updated settings
 */
export async function syncSettings(force = false) {
    console.log(`[settings-sync] Syncing settings (force: ${force})`);

    try {
        // Check if we need to sync (skip if synced within last hour unless forced)
        if (!force && settingsCache.loadTime) {
            const timeSinceLastSync = Date.now() - new Date(settingsCache.loadTime).getTime();
            const ONE_HOUR = 60 * 60 * 1000;

            if (timeSinceLastSync < ONE_HOUR) {
                CONFIG.log(`Skipping sync, last sync ${Math.round(timeSinceLastSync / 1000 / 60)} minutes ago`);
                return settingsCache.domainSettings || {};
            }
        }

        // Fetch from backend
        const settings = await fetchSettingsFromBackend();

        if (settings && settings.allowedSuffixes && Array.isArray(settings.allowedSuffixes)) {
            // Save to storage
            await saveSettingsToStorage(settings);

            // Update cache
            settingsCache.domainSettings = settings;
            settingsCache.loadTime = new Date(settings.lastUpdated);

            CONFIG.log("Settings synced and saved to storage", {
                version: settings.version,
                suffixCount: settings.allowedSuffixes.length
            });

            return settings;
        }

        CONFIG.warn("Sync completed but returned invalid settings");
        return settingsCache.domainSettings || {};

    } catch (error) {
        CONFIG.error("Failed to sync settings", error);
        // Return cached settings if available
        return settingsCache.domainSettings || {};
    }
}

/**
 * Get current settings from storage
 * @returns {Promise<Object>} Current settings
 */
export async function getSettings() {
    console.log("[settings-sync] Getting settings from storage");

    try {
        const settings = await getSettingsFromStorage();

        if (settings) {
            return settings;
        }

        // No settings in storage, try to sync
        CONFIG.log("No settings in storage, triggering sync");
        return await syncSettings(true);

    } catch (error) {
        CONFIG.error("Failed to get settings", error);
        return {};
    }
}

/**
 * Save settings to local storage
 * @param {Object} settings - Settings to save
 * @returns {Promise<void>}
 */
export async function saveSettings(settings) {
    console.log("[settings-sync] Saving settings to storage");

    try {
        await saveSettingsToStorage(settings);
        settingsCache.domainSettings = settings;
        settingsCache.loadTime = new Date();
    } catch (error) {
        CONFIG.error("Failed to save settings", error);
    }
}

/**
 * Clear settings from storage and cache
 * @returns {Promise<void>}
 */
export async function clearSettings() {
    console.log("[settings-sync] Clearing settings");

    return new Promise((resolve) => {
        chrome.storage.local.remove(['domainSettings'], () => {
            settingsCache.domainSettings = null;
            settingsCache.loadTime = null;
            resolve();
        });
    });
}
